/*!
 * \file
 * \brief Error handling functions - source file
 * \author Tobias Ringstrom and Adam Piatyszek
 *
 * -------------------------------------------------------------------------
 *
 * Copyright (C) 1995-2010  (see AUTHORS file for a list of contributors)
 *
 * This file is part of IT++ - a C++ library of mathematical, signal
 * processing, speech processing, and communications classes and functions.
 *
 * IT++ is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * IT++ is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with IT++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * -------------------------------------------------------------------------
 */

#include <iostream>
#include <stdexcept>
#include <cstdlib>

#include "_sci_assert.hpp"

namespace itpp
{

//! \cond
static bool warnings_enabled = true;
static bool file_line_info_enabled = true;
static std::ostream *warn = &std::cerr;
//! \endcond

void itpp_sci_assert_f(std::string ass, std::string msg, std::string file, int line)
{
  std::ostringstream error;
  if(file_line_info_enabled)
  {
    error << "*** Assertion failed in " << file << " on line " << line
          << ":\n" << msg << " (" << ass << ")\n";
  }
  else
  {
    error << msg << " (" << ass << ")\n";
  }
  std::cerr << error.str() << std::flush;
#ifdef ITPP_EXCEPTIONS
  throw std::runtime_error(error.str());
#elif defined( SCI_EXCEPTIONS)
  throw itpp_sci_assert_exception(msg);
#else
  abort();
#endif
}

void itpp_sci_error_f(std::string msg, std::string file, int line)
{
  std::ostringstream error;
  if(file_line_info_enabled)
  {
    error << "*** Error in " << file << " on line " << line << ":\n"
          << msg << "\n";
  }
  else
  {
    error << msg << "\n";
  }
  std::cerr << error.str() << std::flush;
#ifdef ITPP_EXCEPTIONS
  throw std::runtime_error(error.str());
#elif defined( SCI_EXCEPTIONS)
  throw itpp_sci_assert_exception(msg);
#else
  abort();
#endif
}

void itpp_sci_info_f(std::string msg)
{
  std::cerr << msg << std::flush;
}

void itpp_sci_warning_f(std::string msg, std::string file, int line)
{
  if(warnings_enabled)
  {
    if(file_line_info_enabled)
    {
      (*warn) << "*** Warning in " << file << " on line " << line << ":\n"
              << msg << std::endl << std::flush;
    }
    else
    {
      (*warn) << msg << std::endl << std::flush;
    }
  }
}

void itpp_sci_enable_warnings()
{
  warnings_enabled = true;
}

void itpp_sci_disable_warnings()
{
  warnings_enabled = false;
}

void itpp_sci_redirect_warnings(std::ostream *warn_stream)
{
  warn = warn_stream;
}

void itpp_sci_error_msg_style(itpp_sci_error_msg_style_t style)
{
  switch(style)
  {
    case msg_style_full:
      file_line_info_enabled = true;
      break;
    case msg_style_minimum:
      file_line_info_enabled = false;
      break;
    default:
      file_line_info_enabled = true;
  }
}

} //namespace itpp
