/*!
* \brief  sci_bert test application
* \author maki
*/

// #define DEBUG_LEVEL 2

// ---- MSVC  ----
#include <iostream>

// ---- ITPP  ----

// ----- SCI -----
#include "sci\_sci_macros.hpp"
#include "sci\_sci_exception.hpp"
#include "_sci_assert.hpp"

#include "__debug_level.h"
#include "debug.hpp"


using namespace std;
using namespace itpp;
using namespace SCI;

int tf_sci_bert()
{

#ifdef _SCI_
  sci_base *p_bert1, *p_prbs1;
#else
  void *p_bert1, *p_prbs1;
#endif

  int N, W, L, PRBS, fsm;
  int k, K_RELOAD, K_PRESET, K;
  bvec one = bvec("1");
  bvec ce;
  ivec T;
  bmat p, pmat, errmat;
  bvec x, y, z, z1;
  bvec x_ref, y_ref, z_ref;
  ivec state;
  ivec cnt, acc_cnt;
  ivec cnt_ref, acc_cnt_ref;

  int err = 0;
  int sci_type;

  cout << "*** TEST *** " << SCI_NAMESPACE << "::sci_bert" << endl;
  try
  {
    // Symbol size QPSK
    W = 2;
    // test patterns by prbs generator M=2^PRBS-1
    PRBS = 4;
    // BERT correlator - number of symbols
    N = 10;
    // BERT correlator length
    L = W * N;
    // T= [T0, T1, T2] - SYNC for metrics < T1 i.e. BER < 4/20 (more than 1.0E-1);
    T = "2 4 8";

    p_prbs1 = SCI_CREATE(SCI_LFSR);
    sci_type = SCI_GET<int>(p_prbs1, SCI_TYPE);
    itpp_sci_assert(sci_type == SCI_LFSR, "sci_type != SCI_LFSR");

    p_bert1 = SCI_CREATE(SCI_BERT);
    sci_type = SCI_GET<int>(p_bert1, SCI_TYPE);
    itpp_sci_assert(sci_type == SCI_BERT, "sci_type != SCI_BERT");
    // set LFSR as prbs generator
    SCI_SET(p_prbs1, SCI_PRBS, PRBS);
    debug(1) << "SCI::prbs1 PRBS=" << PRBS << endl;
    SCI_SET(p_prbs1, SCI_SYMBOL_SIZE, W);
    debug(1) << "SCI::prbs1 symbol_size W=" << W << endl;
    // set BERT prbs generator
    SCI_SET(p_bert1, SCI_PRBS, PRBS);
    debug(1) << "SCI::bert1 PRBS=" << PRBS << endl;
    // L must be set first - symbol_size <= L
    SCI_SET(p_bert1, SCI_LENGTH, L);
    debug(1) << "SCI::bert1 L=" << L << endl;
    SCI_SET(p_bert1, SCI_SYMBOL_SIZE, W);
    debug(1) << "SCI::bert1 W=" << W << endl;
    SCI_SET(p_bert1, SCI_THRESHOLD, T);
    debug(1) << "SCI::bert1 T=" << T << endl;
    // set initiation state
    SCI_SET(p_bert1, SCI_FSM, (int)BERT_FSM_RELOAD);
    cout << "SCI::bert1 FSM(RELOAD)" << endl;

    // number of symbols to transition from RELOAD->PRESET
    K_RELOAD = SCI_GET<int>(p_prbs1, SCI_LENGTH) / W;
    debug(1) << "RELOAD->PRESET needs K_RELOAD = " << K_RELOAD << " symbols" << endl;
    for(k = 0; k < K_RELOAD; k++)
    {
      debug(2) << "[" << k << "]" << endl;
      p = SCI_GEN<bmat>(p_prbs1, one);
      debug(2) << " prbs =" << p << endl;
      state = SCI_PROC<ivec>(p_bert1, one, p);
      SCI_SET(p_bert1, SCI_ADR, (int)BERT_ADR_LSRX);
      x = SCI_GET<bvec>(p_bert1, SCI_STATE);
      SCI_SET(p_bert1, SCI_ADR, (int)BERT_ADR_LSRY);
      y = SCI_GET<bvec>(p_bert1, SCI_STATE);
      SCI_SET(p_bert1, SCI_ADR, (int)BERT_ADR_PRBS);
      z = SCI_GET<bvec>(p_bert1, SCI_STATE);
      fsm = SCI_GET<int>(p_bert1, SCI_FSM);
      z1 = SCI_GET<bvec>(p_prbs1, SCI_STATE);
      debug(2) << " x=" << x << endl;
      debug(2) << " y=" << y << endl;
      debug(2) << " z=" << z << endl;
      debug(2) << "z1=" << z1 << endl;
      debug(2) << " metrics=" << SCI_GET<int>(p_bert1, SCI_METRICS) << endl;
      debug(2) << " state =" << fsm << endl;
      debug(2) << " fsm =" << fsm << endl;
    }
    x_ref = bvec("1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0");
    debug(1) << "    x=" << x << endl;
    debug(1) << "x_ref=" << x_ref << endl;
    y_ref = bvec("0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0");
    debug(1) << "    y=" << y << endl;
    debug(1) << "y_ref=" << y_ref << endl;
    z_ref = bvec("0 0 1 1");
    debug(1) << "    z=" << z << endl;
    debug(1) << "z_ref=" << z_ref << endl;
    debug(1) << "  fsm=" << fsm << endl;

    itpp_sci_assert(x == x_ref, "x!=x_ref");
    itpp_sci_assert(y == y_ref, "y!=y_ref");
    itpp_sci_assert(z == z_ref, "z!=z_ref");
    itpp_sci_assert(fsm == BERT_FSM_PRESET, "fsm != BERT_FSM_PRESET");

    // number of symbols to transition from RELOAD->PRESET
    K_PRESET = SCI_GET<int>(p_bert1, SCI_LENGTH) / W;
    debug(1) << "RELOAD->PRESET needs K_RELOAD = " << K_RELOAD << " symbols" << endl;
    for(k = 0; k < K_PRESET; k++)
    {
      debug(2) << "[" << k << "]" << endl;
      p = SCI_GEN<bmat>(p_prbs1, one);
      debug(2) << " prbs =" << p << endl;
      state = SCI_PROC<ivec>(p_bert1, one, p);
      SCI_SET(p_bert1, SCI_ADR, (int)BERT_ADR_LSRX);
      x = SCI_GET<bvec>(p_bert1, SCI_STATE);
      SCI_SET(p_bert1, SCI_ADR, (int)BERT_ADR_LSRY);
      y = SCI_GET<bvec>(p_bert1, SCI_STATE);
      SCI_SET(p_bert1, SCI_ADR, (int)BERT_ADR_PRBS);
      z = SCI_GET<bvec>(p_bert1, SCI_STATE);
      fsm = SCI_GET<int>(p_bert1, SCI_FSM);
      z1 = SCI_GET<bvec>(p_prbs1, SCI_STATE);
      debug(2) << " x=" << x << endl;
      debug(2) << " y=" << y << endl;
      debug(2) << " z=" << z << endl;
      debug(2) << "z1=" << z1 << endl;
      debug(2) << " metrics=" << SCI_GET<int>(p_bert1, SCI_METRICS) << endl;
      debug(2) << " state =" << fsm << endl;
      debug(2) << " fsm =" << fsm << endl;
    }
    x_ref = bvec("1 1 0 1 0 1 1 0 0 1 0 0 0 1 1 1 1 0 1 0");
    debug(1) << "    x=" << x << endl;
    debug(1) << "x_ref=" << x_ref << endl;
    y_ref = bvec("1 1 0 1 0 1 1 0 0 1 0 0 0 1 1 1 1 0 1 0");
    debug(1) << "    y=" << y << endl;
    debug(1) << "y_ref=" << y_ref << endl;
    z_ref = bvec("1 0 1 1");
    debug(1) << "    z=" << z << endl;
    debug(1) << "z_ref=" << z_ref << endl;
    debug(1) << "  fsm=" << fsm << endl;
    itpp_sci_assert(x == x_ref, "x!=x_ref");
    itpp_sci_assert(y == y_ref, "y!=y_ref");
    itpp_sci_assert(z == z_ref, "z!=z_ref");
    itpp_sci_assert(fsm == BERT_FSM_SYNC, "fsm != BERT_FSM_SYNC");

    cout << "Locking" << endl;
    acc_cnt = SCI_GET<ivec>(p_bert1, SCI_ACC_CNT);
    cnt = SCI_GET<ivec>(p_bert1, SCI_CNT);
    acc_cnt_ref = ivec("0 0");
    cnt_ref = ivec("0 0");
    debug(1) << "         cnt=" << cnt << endl;
    debug(1) << "     cnt_ref=" << cnt_ref << endl;
    debug(1) << "     acc_cnt=" << acc_cnt << endl;
    debug(1) << " acc_cnt_ref=" << acc_cnt_ref << endl;
    itpp_sci_assert(cnt == cnt_ref, "cnt != cnt_ref");
    itpp_sci_assert(acc_cnt == acc_cnt_ref, "acc_cnt != acc_cnt_ref");

    // test BER
    K = 10;
    debug(1) << "SYNC with no errors K = " << K << " symbols" << endl;
    ce = ones_b(K);
    pmat = SCI_GEN<bmat>(p_prbs1, ce); // must generate continously - this is what bert expects
    SCI_PROC<ivec>(p_bert1, ce, pmat);
    acc_cnt = SCI_GET<ivec>(p_bert1, SCI_ACC_CNT);
    cnt = SCI_GET<ivec>(p_bert1, SCI_CNT);
    acc_cnt_ref = ivec("0 20");
    cnt_ref = ivec("0 20");
    debug(1) << "         fsm=" << fsm << endl;
    debug(1) << "         cnt=" << cnt << endl;
    debug(1) << "     cnt_ref=" << cnt_ref << endl;
    debug(1) << "     acc_cnt=" << acc_cnt << endl;
    debug(1) << " acc_cnt_ref=" << acc_cnt_ref << endl;
    itpp_sci_assert(cnt == cnt_ref, "cnt != cnt_ref");
    itpp_sci_assert(acc_cnt == acc_cnt_ref, "acc_cnt != acc_cnt_ref");
    itpp_sci_assert(fsm == BERT_FSM_SYNC, "fsm != BERT_FSM_SYNC");

    // make one err
    debug(1) << "SYNC with 1 error K = " << K << " symbols" << endl;
    errmat = zeros_b(K, W);
    errmat(0, 0) = 1;
    pmat = SCI_GEN<bmat>(p_prbs1, ce); // must generate continously - this is what bert expects
    pmat = pmat + errmat;
    SCI_PROC<ivec>(p_bert1, ce, pmat);

    acc_cnt = SCI_GET<ivec>(p_bert1, SCI_ACC_CNT);
    cnt = SCI_GET<ivec>(p_bert1, SCI_CNT);

    acc_cnt_ref = ivec("1 40");
    cnt_ref = ivec("1 20");
    debug(1) << "        fsm=" << fsm << endl;
    debug(1) << "        cnt=" << cnt << endl;
    debug(1) << "    cnt_ref=" << cnt_ref << endl;
    debug(1) << "    acc_cnt=" << acc_cnt << endl;
    debug(1) << " acc_cntref=" << acc_cnt_ref << endl;
    itpp_sci_assert(cnt == cnt_ref, "cnt != cnt_ref");
    itpp_sci_assert(acc_cnt == acc_cnt_ref, "acc_cnt != acc_cnt_ref");
    itpp_sci_assert(fsm == BERT_FSM_SYNC, "fsm != BERT_FSM_SYNC");

    SCI_DESTROY(p_prbs1);
    SCI_DESTROY(p_bert1);
  }
  catch(itpp_sci_assert_exception &except)
  {
    cout << "\n itpp_sci_assert_exception:" << endl << except.get_msg() << ":" << except.get_info() << endl;
    err = -1;
  }
  catch(sci_exception &except)
  {
    cout << "\n sci exception:" << endl << except.get_msg() << ":" << except.get_info() << endl;
    err = -2;
  }
  catch(...)
  {
    cout << "\n unknown exception ???" << endl;
    err = -3;
  }
  return err;
}
