/*!
* \brief - binbuff test application
* \author maki
*/

// ---- MSVC  ----
#include <iostream>
#include <iomanip>

// ---- ITPP  ----
#include "itpp/itbase.h"

// ----- SIM -----
#include "sim\_sim_lib.hpp"
#include "sim\_sim_exception.hpp"

// ----- SCI -----
#include "_sci_assert.hpp"

#include "__debug_level.h"
#include "debug.hpp"

using namespace std;
using namespace itpp;
using namespace SIM;

int tf_sim_binbuff()
{
  SIM::binbuff bb1;  

  const int W  = 8;   // input symbols width = bytes 
  const int N  = 4;   // number of input symbols

  const int WA = 4;   // testA output symbols width 
  const int WB = 16;  // testB output symbols width

  const int NA = int((N * W) / WA);       // process NA symbols of WA width
  const int NB = int((2 * N * W) / WB);   // generate NB symbols of WB width

  bvec ce, ce0, ce1;
  bmat ceio, yA, yB;

  // 4 bytes 
  const bmat     x1 = bmat("1 1 1 1 0 0 0 0; 1 1 1 1 0 0 0 1; 1 1 1 1 0 0 1 0; 1 1 1 1 0 0 1 1");
  // 8  4bit symbols
  const bmat yA_ref = bmat("1 1 1 1; 0 0 0 0; 1 1 1 1; 0 0 0 1; 1 1 1 1; 0 0 1 0; 1 1 1 1; 0 0 1 1");
  // 4  16bit symbols
  const bmat yB_ref = bmat("1 1 1 1 0 0 0 0 1 1 1 1 0 0 0 1; 1 1 1 1 0 0 1 0 1 1 1 1 0 0 1 1;\
                            1 1 1 1 0 0 0 0 1 1 1 1 0 0 0 1; 1 1 1 1 0 0 1 0 1 1 1 1 0 0 1 1");

  int err = 0;

  cout << "*** TEST SIM *** binbuff " << endl;
  try
  {
    ce.set_length(N);
    ce.ones();

    debug(1) << "x1 = " << endl << x1 << endl;

    // buffer size in bits
    bb1.set_size(W * N);
    // symbols symbol_size 
    bb1.set_symbol_size(W);

    //[ceio]=[1|0]
    // ceio for binbuuf
    ce1 = to_bvec(to_ivec(ones(N)));
    ce0 = to_bvec(to_ivec(zeros(N)));
    // push in all N bytes - ceio=[cei|cei] = [1|0] - filling -up - ceo=0
    ceio.set_size(N, 2);
    ceio.set_col(0, ce1);
    ceio.set_col(1, ce0);
    bb1.process(ceio, x1);
    debug(1) << "nrof_elements =" << bb1.nrof_elements() << endl;
    itpp_sci_assert(bb1.nrof_elements() == N*W, "nrof_elements() != N*W");

    // pop in all NA symbols of WA width by processing with cei=0 - empting the buffer 
    // ceio[NA,:]=[0,1]
    ce1 = to_bvec(to_ivec(ones(NA)));
    ce0 = to_bvec(to_ivec(zeros(NA)));
    ceio.set_size(NA, 2);
    ceio.set_col(0, ce0);
    ceio.set_col(1, ce1);
    // symbols new symbol_size -  convert all bits into new symbols
    bb1.set_symbol_size(WA);
    yA = bb1.process(ceio, ceio);
    debug(1) << "yA =" << endl << yA << endl;
    itpp_sci_assert(yA == yA_ref, "yA != yA_ref");

    debug(1) << "nrof_elements =" << bb1.nrof_elements() << endl;
    itpp_sci_assert(bb1.nrof_elements() == 0, "nrof_elements() != 0");

    // put resized data - generate new symbols
    ce1 = to_bvec(to_ivec(ones(NB)));
    bb1.put(rvectorize(x1));
    debug(1) << "nrof_elements =" << bb1.nrof_elements() << endl;
    itpp_sci_assert(bb1.nrof_elements() == N*W, "nrof_elements() != N*W");

    bb1.set_symbol_size(WB);
    yB = bb1.generate(ce1);
    debug(1) << "yB =" << endl << yB << endl;
    itpp_sci_assert(yB == yB_ref, "yB != yB_ref");
    debug(1) << "nrof_elements =" << bb1.nrof_elements() << endl;
    itpp_sci_assert(bb1.nrof_elements() == N*W, "nrof_elements() != N*W");

  }
  catch (itpp_sci_assert_exception &except)
  {
    cout << "\n itpp_sci_assert_exception:" << endl << except.get_msg() << ":" << except.get_info() << endl;
    err = -1;
  }
  catch (sim_exception &except)
  {
    cout << "\n sim_exception:" << endl << except.get_msg() << ":" << except.get_info() << endl;
    err = -2;
  }
  catch (...)
  {
    cout << "\n unknown exception ???" << endl;
    err = -3;
  }
  return err;

}

