/*!
* \brief - cofdm test application
* \author maki
*/

// ---- MSVC  ----
#include <iostream>
#include <complex>
#include <iomanip>

// ---- ITPP  ----
#include "itpp/itbase.h"

// ----- SIM -----
#include "sim\_sim_lib.hpp"
#include "sim\_sim_exception.hpp"

// ----- SCI -----
#include "_sci_assert.hpp"

#include "__debug_level.h"
#include "debug.hpp"

using namespace std;
using namespace itpp;
using namespace SIM;

int tf_sim_cofdm()
{

  SIM::cofdm_mod    cofdm_mod1;
  SIM::cofdm_dem    cofdm_dem1;
  SIM::cofdm_map    cofdm_map1;
  SIM::cofdm_demap  cofdm_demap1;
  SIM::cofdm_sel    cofdm_sel1;

  const int NFFT = 16; // NFFT
  const int CP = 4;    // CP 
  const int M = 4;     // QPSK

  cvec y_map, y_mod, y_dem;
  cmat m_map, m_mod, m_dem;

  ivec y_demap_data;
  bvec y_demap_pilots;
  cvec z_mod, z_dem;        // zero carriers  
  cvec cp_lead;             // CP = x[0:CP-1]
  cvec cp_trail;            // CP = x[NFFT:NFFT+CP-1]
  bvec true_vec;
  bvec ce;

  double epsilon = 1.0E-6;
  int err = 0;

  cout << "*** TEST SIM *** cofdm " << endl;  
  try
  {
    debug(1) <<  "NFFT = " << NFFT << endl;
    debug(1) << " CP = " << CP << endl;
    debug(1) << " M = " << M << endl;
    // set modulator
    cofdm_mod1.set_NFFT(NFFT);
    cofdm_mod1.set_CP(CP);
    // set demodulator
    cofdm_dem1.set_NFFT(NFFT);
    cofdm_dem1.set_CP(CP);
    // set mapper
    cofdm_map1.set_NFFT(NFFT);
    cofdm_map1.set_qam_size(M);
    // set demapper
    cofdm_demap1.set_NFFT(NFFT);
    cofdm_demap1.set_qam_size(M);
    // set sel
    cofdm_sel1.set_NFFT(NFFT);
    // z - zero carrier
    // p - pilot carrier
    // d - data carrier
    // 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15
    // z p d d d p d d d p d  d  d  p  d  z
    ivec dcindx = " 2 3 4 6 7 8 10 11 12 14 ";
    ivec dc     = " 0 1 2 3 0 1 2  3  0  1  ";
    ivec zcindx = " 0 15 ";
    ivec pcindx = " 1 5 9 13";
    bvec pc     = " 0 1 0 1";
    debug(1) << " data_carriers dcindx = " << endl << dcindx << endl;
    debug(1) << " pilots_carriers pcindx = " << endl << pcindx << endl;
    debug(1) << " zero_carriers zcindx = " << endl << zcindx << endl;
    debug(1) << " data dc = " << endl << dc << endl;
    debug(1) << " pilots pc = " << endl << pc << endl;

    z_mod = to_cvec(zeros(zcindx.length()), zeros(zcindx.length()));
    true_vec = ones_b(zcindx.length());

    // set data pilots zero carrier indexes for map
    cofdm_map1.set_data_carriers(dcindx);
    cofdm_map1.set_pilots_carriers(pcindx);
    cofdm_map1.set_zero_carriers(zcindx);
    // set data pilots zero carrier indexes for demap
    cofdm_demap1.set_data_carriers(dcindx);
    cofdm_demap1.set_pilots_carriers(pcindx);
    cofdm_demap1.set_zero_carriers(zcindx);
    // set selected carriers for zero-carriers
    cofdm_sel1.set_carriers_selector(zcindx);

    // test immediate interface 
    // map data/pilots into symbols
    cofdm_map1.set_data(dc);
    cofdm_map1.set_pilots(pc);
    y_map = cofdm_map1.get_carriers();
    debug(1) << "cofdm_map1.get_output() y_map = " << endl << y_map << endl;
    // modulate
    cofdm_mod1.set_carriers(y_map);
    y_mod = cofdm_mod1.get_symbol();
    debug(1) << "cofdm_mod1.get_output() y_mod = " << endl << y_mod << endl;
    
    cp_lead = y_mod.left(CP);         // CP = x[0:CP-1]
    cp_trail = y_mod.right(CP);       // CP = x[NFFT:NFFT+CP-1]
    itpp_sci_assert(cp_lead == cp_trail, "cp_lead != cp_trail");

    // demodulate
    cofdm_dem1.set_symbol(y_mod);
    y_dem = cofdm_dem1.get_carriers();
    debug(1) << "cofdm_dem1.get_output() y_dem = " << endl << y_dem << endl;
    // demap
    cofdm_demap1.set_carriers(y_dem);
    y_demap_data = cofdm_demap1.get_data();
    debug(1) << "cofdm_demap1.get_data() y_demap_data = " << endl << y_demap_data << endl;
    y_demap_pilots = cofdm_demap1.get_pilots();
    debug(1) << "cofdm_demap1.get_pilots() y_demap_pilots = " << endl << y_demap_pilots << endl;
    
    // select zero carriers
    cofdm_sel1.set_carriers(y_dem);
    z_dem = cofdm_sel1.get_selected_carriers();
    debug(1) << "cofdm_sel1.get_sel() z_demod = " << endl << z_dem << endl;

    itpp_sci_assert(y_demap_data == dc, "y_demap_data != dc");
    itpp_sci_assert(y_demap_pilots == pc, "y_demap_pilots != pc");        
    itpp_sci_assert((abs(z_dem - z_mod) < epsilon) == true_vec, "z_dem != z_mod");

    // test clocked interface with double tick
    ce = ones_b(2);
    imat m_pd, m_pd_dem;
    m_pd.set_size(2, pc.length() + dc.length());
    m_pd.set_row(0, concat(reverse(to_ivec(pc)), reverse(dc)));
    m_pd.set_row(1, concat(to_ivec(pc), dc));

    debug(1) << "m_pd = " << endl << m_pd << endl;
    m_map = cofdm_map1.process(ce, m_pd);
    
    debug(1) << "my_map = " << endl << m_map << endl;
    itpp_sci_assert((abs(m_map.get_row(1) - y_map) < epsilon) == ones_b(NFFT), "my_map(1) != y_map");

    m_mod = cofdm_mod1.process(ce, m_map);
    m_dem = cofdm_dem1.process(ce, m_mod);
    m_pd_dem = cofdm_demap1.process(ce, m_dem);

    debug(1) << "m_pd_dem = " << endl << m_pd_dem << endl;
    itpp_sci_assert( m_pd_dem == m_pd, "m_pd_dem != m_pd");

  }
  catch (itpp_sci_assert_exception &except)
  {
    cout << "\n itpp_sci_assert_exception:" << endl << except.get_msg() << ":" << except.get_info() << endl;
    err = -1;
  }
  catch (sim_exception &except)
  {
    cout << "\n sim_exception:" << endl << except.get_msg() << ":" << except.get_info() << endl;
    err = -2;
  }
  catch (...)
  {
    cout << "\n unknown exception ???" << endl;
    err = -3;
  }
  return err;
}

