/*!
* \brief - sim_lfsr test application
* \author maki
*/

// ---- MSVC  ----
#include <iostream>
#include <complex>
#include <iomanip>

// ---- ITPP  ----
#include "itpp/itbase.h"

// ----- SIM -----
#include "sim\_sim_lib.hpp"
#include "sim\_sim_exception.hpp"

// ----- SCI -----
#include "_sci_assert.hpp"

#include "__debug_level.h"
#include "debug.hpp"

using namespace std;
using namespace itpp;
using namespace SIM;


int tf_sim_lfsr()
{
  SIM::lfsr lfsr1;

  int N, M, W, L, T, i;
  bvec c0, c1, z0, z1;
  bvec ce;
  bmat y, yref;
  int err = 0;

  cout << "*** TEST SIM *** lsfr"  << endl;
  try
  {
    // It looks like the implementation is like J.83B standard PRBS randomizer
    // PRBS M = 4, sequence lenght =  2^4-1
    //       1+X3+X4
    //       0 1 2 3 4
    c0 = bvec("1 0 0 1 1");
    L = c0.length() - 1;
    M = ((int)pow(2.0, L) - 1);  // prbs sequence lenght
    W = M;

    // minimal setup set_taps()/prbs() and set_symbol_size()
    debug(1) << "PRBS sequence lenght M =" << M << endl;
    debug(1) << "c0 = " << c0 << endl;
    lfsr1.set_poly(c0);
    debug(1) << "lfsr1.set_symbol_size() W =" << W << "== PRBS sequence" << endl;
    lfsr1.set_symbol_size(W);

    c1 = lfsr1.get_poly();
    debug(1) << "c1 = " << c0 << endl;
    itpp_sci_assert(c1 == c0, "c1 != c0");
    // reset done via set_poly
    // state 1 is translated to [001] when reverse_flag is set (default)
    // because convention for lfsr poly =[g0,g1,g2,g3,]
    // for state s0 = [s1,s2,s3,..,]  (no s^0 !)
    debug(1) << "revese flag =" << lfsr1.get_rev_flag() << endl;

    z0.set_length(L);
    z0 = dec2bin(L, 1);
    z1 = lfsr1.get_state();
    debug(1) << "z0 = " << z0 << endl;
    debug(1) << "z1 = " << z1 << endl;
    itpp_sci_assert(z1 == z0, "z1 != z0");

    T = lfsr1.get_symbol_size();
    debug(1) << "lfsr1.get_symbol_size() T =" << T << endl;
    itpp_sci_assert(T == W, "T != W");
    // number of ticks == number of symbols
    N = 4;
    ce.set_length(N);
    ce.ones();
    // one symbol is whole PRBS sequence, so you should get N identical symbols
    yref.set_size(N, W);
    debug(1) << "number_of ticks == number of symbols N =" << N << endl;

    for(i = 0; i < N; i++)
    {
      yref.set_row(i, reverse(bvec("0 0 1 1 0 1 0 1 1 1 1 0 0 0 1")));
    }
    debug(1) << "yref =" << endl  << yref << endl;

    y = lfsr1.generate(ce);
    debug(1) << "y =" << endl << y << endl;
    itpp_sci_assert(y == yref, "y != yref");

    lfsr1.set_rev_flag(false);
    debug(1) << "revese flag =" << lfsr1.get_rev_flag() << endl;
    itpp_sci_assert(!lfsr1.get_rev_flag(), "lfsr1.get_rev_flag() != false");

    for(i = 0; i < N; i++)
    {
      yref.set_row(i, bvec("0 0 1 1 0 1 0 1 1 1 1 0 0 0 1"));
    }
    debug(1) << "yref =" << endl << yref << endl;
    y = lfsr1.generate(ce);
    debug(1) << "y =" << endl << y << endl;
    itpp_sci_assert(y == yref, "y != yref");
  }
  catch(itpp_sci_assert_exception &except)
  {
    cout << "\n itpp_sci_assert_exception:" << endl << except.get_msg() << ":" << except.get_info() << endl;
    err = -1;
  }
  catch(sim_exception &except)
  {
    cout << "\n sim_exception:" << endl << except.get_msg() << ":" << except.get_info() << endl;
    err = -2;
  }
  catch(...)
  {
    cout << "\n unknown exception ???" << endl;
    err = -3;
  }
  return err;

}

