/*!
* \brief - sci_src_x test application
* \author maki
*/

// ---- MSVC  ----
#include <iostream>
#include <complex>
#include <iomanip>

// ---- ITPP  ----
#include "itpp/itbase.h"

// ----- SIM -----
#include "sim\_sim_lib.hpp"
#include "sim\_sim_exception.hpp"

// ----- SCI -----
#include "_sci_assert.hpp"

#include "__debug_level.h"
#include "debug.hpp"

using namespace std;
using namespace itpp;
using namespace SIM;

static const int NTAPS = 9;
static const int WIN = WIN_HANNING;

static double sinc_pi(double x)
{
  double y;

  if (x == 0.0)
  {
    y = 1.0;
  }
  else
  {
    y = sin(pi * x) / (pi * x);
  }
  return (y);
}

// calculate casual sinc response for nt - normalised fractional time 
static double h(double nt)
{
  double y, yw;
  double dTN, TN;
  double  w;
  double scale_cor;


  TN = (double)(NTAPS - 1);       // normalised impulse response time (N-1)
  dTN = TN / 2;                   // casual delay (N-1)/2
  if ((nt < TN) && (nt >= 0.0))
  {
    y = sinc_pi(nt - dTN);
  }
  else
  {
    y = 0.0;
  }

  switch (WIN)
  {
    case WIN_HANNING: // Hanning
      w = 0.5 * (1 - cos(2 * pi * nt / TN));
      break;
    case WIN_KAISER_BESSEL: // Kaiser-Besell
      w = (1 - 1.24 * cos(2 * pi * nt / TN) + 0.244 * cos(4 * pi * nt / TN) - 0.00305 * cos(6 * pi * nt / TN)) / 2.48;
      scale_cor = 1 / 1.002843;
      w = w * scale_cor;
      break;
    case WIN_FLAT_TOP:// Flat-Top
      w = (1 - 1.93 * cos(2 * pi * nt / TN) + 1.29 * cos(4 * pi * nt / TN) - 0.388 * cos(6 * pi * nt / TN) - 0.0322 * cos(8 * pi * nt / TN)) / 4.46;
      scale_cor = 1 / 1.0259641;
      w = w * scale_cor;
      break;
    default: // 0 and others rectangular
      w = 1.0;
      break;
  }
  yw = y * w;
  return(yw);
}

static vec h_vec(const vec &t)
{
  vec y;
  int K,k;
  
  K = t.size();
  y.set_length(K);
  for (k = 0; k < K; k++)
  {
    y[k] = h(t[k]);
  }
  return y;
}


int tf_sim_src_x()
{
  SIM::src_x	src_x1;

  const int N = NTAPS * 2;
  const double TAU = -0.1;

  bvec cei, ceo, true_vec;
  bmat ceio;
  vec  x, t, tau, h_ref;
  cvec x_x1, t_x1;
  cmat  xt_x1;
  cvec y_x1, y_x1_ref;

  const double epsilon = 1.0E-6;
  int err = 0;

  cout << "*** TEST SIM *** src_x " << endl;
  try
  {

    cout << fixed << setw(10) << setprecision(7);

    debug(1) << "TAU  = " << TAU << endl;

    src_x1.set_size(NTAPS);
    src_x1.get_size();
    debug(1) <<  "src_x1.get_size  = "  <<  src_x1.get_size() << endl;
    itpp_sci_assert(src_x1.get_size() == NTAPS, "src_x1.get_size() != NTAPS");

    t = linspace(0, N - 1, N);        // [0,1,2...] normalized sampling time 
    tau = TAU * ones(N);              // constant fractional resampling  offset
    t = t + tau;                      // re-sample time instances

    cei = ones_b(N);
    ceo = ones_b(N);
    true_vec = ones_b(N);

    ceio.set_size(N, 2);
    ceio.set_col(0, cei);
    ceio.set_col(1, ceo);
    debug(3) << "ceio  = \n" <<	ceio << endl;

    // x_x1 is to  get 1+j1 - 2 impulse responses
    x = zeros(N);
    x(0) = 1.0;
    x_x1 = to_cvec(x, x);
    // tau to get interpolation
    t_x1 = to_cvec(tau, zeros(N));
    // cmat [xi+j*xq|tau+j*0]
    xt_x1.set_size(N, 2);
    xt_x1.set_col(0, x_x1);
    xt_x1.set_col(1, t_x1);
    debug(3) << "xt_x1  = \n" << 	xt_x1 << endl;

    // get impulse response
    y_x1 = src_x1.process(ceio, xt_x1);
    // get reference - casual sinc response for noramlized re-sampling time 
    h_ref = h_vec(t);
    // both filters should respond with impulse response
    y_x1_ref = to_cvec(h_ref, h_ref);
    debug(1) << "y_x1  = \n"  <<  y_x1   << endl;
    debug(1) << "y_x1_ref  = \n" << y_x1_ref << endl;

    itpp_sci_assert((abs(y_x1 - y_x1_ref) < epsilon) == true_vec, "y_x1 != y_x1_ref");
  }
  catch (itpp_sci_assert_exception &except)
  {
    cout << "\n itpp_sci_assert_exception:" << endl << except.get_msg() << ":" << except.get_info() << endl;
    err = -1;
  }
  catch (sim_exception &except)
  {
    cout << "\n sim_exception:" << endl << except.get_msg() << ":" << except.get_info() << endl;
    err = -2;
  }
  catch (...)
  {
    cout << "\n unknown exception ???" << endl;
    err = -3;
  }
  return err;
}

