/*!
 * \file
 * \brief Error handling functions - header file
 * \author Tobias Ringstrom and Adam Piatyszek
 *
 * -------------------------------------------------------------------------
 *
 * Copyright (C) 1995-2010  (see AUTHORS file for a list of contributors)
 *
 * This file is part of IT++ - a C++ library of mathematical, signal
 * processing, speech processing, and communications classes and functions.
 *
 * IT++ is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * IT++ is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with IT++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * -------------------------------------------------------------------------
 */

#ifndef ITPP_SCI_ASSERT_H
#define ITPP_SCI_ASSERT_H

#define nITPP_EXCEPTIONS
#define SCI_EXCEPTIONS

#include <sstream>
#include <string>

#include "sci\_sci_exception.hpp"

namespace itpp
{

/*!
  For the following macros, the argument \c s is a string that is displayed.

  \code
  itpp_sci_assert(t,s);           // Abort if t is not true
  itpp_sci_assert_debug(t,s);     // Abort if t is not true and NDEBUG is not defined
  itpp_sci_error_if(t,s);         // Abort if t is true
  itpp_sci_error(s);              // Abort
  itpp_sci_info(s);               // Display a message
  itpp_sci_info_debug(s);         // Display a message if NDEBUG is not defined
  itpp_sci_info_no_endl(s);       // Display a message without appended "std::endl"
  itpp_sci_info_no_endl_debug(s); // Display a message without appended "std::endl" if NDEBUG is not defined
  itpp_sci_warning(s);            // Display a warning
  \endcode

  \c itpp_sci_assert(), \c itpp_sci_error(), \c itpp_sci_error_if(), \c itpp_sci_info(), \c
  itpp_sci_info_no_endl() and \c itpp_sci_warning() are always active, whereas \c
  itpp_sci_assert_debug(), \c itpp_sci_info_debug() and \c itpp_sci_info_no_endl_debug()
  depends on the \c NDEBUG compile time definition. If \c NDEBUG is
  defined, then none of these macros is executed.

  \note \c itpp_sci_assert0() and \c itpp_sci_assert1() macros are still defined for
  backward compatibility, but \c itpp_sci_assert_debug() should be used instead
  of them.
*/

//!@{

//! Helper function for the \c itpp_sci_assert and \c itpp_sci_assert_debug macros
void itpp_sci_assert_f(std::string ass, std::string msg, std::string file, int line);
//! Helper function for the \c itpp_sci_error and \c itpp_sci_error_if macros
void itpp_sci_error_f(std::string msg, std::string file, int line);
//! Helper function for the \c itpp_sci_info and \c itpp_sci_info_debug macros
void itpp_sci_info_f(std::string msg);
//! Helper function for the \c itpp_sci_warning macro
void itpp_sci_warning_f(std::string msg, std::string file, int line);

//! Enable/disable using exceptions for error handling.
void itpp_sci_enable_exceptions(bool on);
//! Enable warnings
void itpp_sci_enable_warnings();
//! Disable warnings
void itpp_sci_disable_warnings();
//! Redirect warnings to the ostream warn_stream
void itpp_sci_redirect_warnings(std::ostream *warn_stream);

//! Style of assert, error and warning messages.
enum itpp_sci_error_msg_style_t { msg_style_full, msg_style_minimum };

//! Set preferred style of assert, error and warning messages
void itpp_sci_error_msg_style(itpp_sci_error_msg_style_t style);


//! Abort if \c t is not true
#define itpp_sci_assert(t,s)      \
  if (!(t)) {       \
    std::ostringstream m_sout;     \
    m_sout << s;      \
    itpp::itpp_sci_assert_f(#t,m_sout.str(),__FILE__,__LINE__); \
  } else       \
    ((void) 0)

#if defined(NDEBUG)
//! Abort if \c t is not true and NDEBUG is not defined
#  define itpp_sci_assert_debug(t,s) ((void) (t))
#else
//! Abort if \c t is not true and NDEBUG is not defined
#  define itpp_sci_assert_debug(t,s) itpp_sci_assert(t,s)
#endif // if defined(NDEBUG)

//! Deprecated macro. Please use \c itpp_sci_assert_debug() instead.
#define itpp_sci_assert0(t,s) itpp_sci_assert_debug(t,s)
//! Deprecated macro. Please use \c itpp_sci_assert_debug() instead.
#define itpp_sci_assert1(t,s) itpp_sci_assert_debug(t,s)


//! Abort if \c t is true
#define itpp_sci_error_if(t,s)    \
  if((t)) {      \
    std::ostringstream m_sout;    \
    m_sout << s;     \
    itpp::itpp_sci_error_f(m_sout.str(),__FILE__,__LINE__); \
  } else      \
    ((void) 0)

//! Abort unconditionally
#define itpp_sci_error(s)     \
  if (true) {      \
    std::ostringstream m_sout;    \
    m_sout << s;     \
    itpp::itpp_sci_error_f(m_sout.str(),__FILE__,__LINE__); \
  } else      \
    ((void) 0)


//! Print information message
#define itpp_sci_info(s)    \
  if (true) {     \
    std::ostringstream m_sout;   \
    m_sout << s << std::endl;   \
    itpp::itpp_sci_info_f(m_sout.str());  \
  } else     \
    ((void) 0)

//! Print information message without \c std::endl at the end
#define itpp_sci_info_no_endl(s)   \
  if (true) {     \
    std::ostringstream m_sout;   \
    m_sout << s;    \
    itpp::itpp_sci_info_f(m_sout.str());  \
  } else     \
    ((void) 0)

#if defined(NDEBUG)
//! Print information message if NDEBUG is not defined
#  define itpp_sci_info_debug(s) ((void) 0)
/*!
  \brief Print information message without \c std::endl at the end if
  NDEBUG is not defined
*/
#  define itpp_sci_info_no_endl_debug(s) ((void) 0)
#else
//! Print information message if NDEBUG is not defined
#  define itpp_sci_info_debug(s) itpp_sci_info(s)
/*!
  \brief Print information message without \c std::endl at the end if
  NDEBUG is not defined
*/
#  define itpp_sci_info_no_endl_debug(s) itpp_sci_info_no_endl(s)
#endif // if defined(NDEBUG)


//! Display a warning message
#define itpp_sci_warning(s)     \
  if (true) {      \
    std::ostringstream m_sout;    \
    m_sout << s;     \
    itpp::itpp_sci_warning_f(m_sout.str(),__FILE__,__LINE__); \
  } else      \
    ((void) 0)

class itpp_sci_assert_exception : public std::exception
{
  int info_;
  std::string msg_;

public:

  int get_info() const
  {
    return info_;
  };
  std::string get_msg()
  {
    return msg_;
  };

  itpp_sci_assert_exception(std::string  msg, int info) : msg_(msg), info_(info)
  {

  };

  itpp_sci_assert_exception(std::string  msg) : msg_(msg)
  {
    info_ = 0;
  };

  itpp_sci_assert_exception()
  {
    msg_ = "Unknown Error";
    info_ = -1;
  };

};



//!@}

} // namespace itpp

#endif // #ifndef ITASSERT_H
