/*xxx
* \brief - bit error rate tester
* \author maki
*/



#ifndef SIM_BERT_HPP
#define SIM_BERT_HPP

#include <itpp/itbase.h>
#include <itpp/comm/sequence.h>

#include "sim\_sim_enum_def.h"
#include "sim\sim_lfsr.hpp"
#include "sim\sim_lsr.hpp"


using namespace itpp;
using namespace std;

namespace SIM
{

/*!
\brief BERT - Bit Error Rate Tester
\n Core function SIM::bert.process() \n
\details
\n For active clock tick ce[i], input symbols x[i,:] are shifted into X LSR and internally generated PRBS symbols y[i,:] are shifted into Y LSR.
\n When \ref BERT_FSM is in SYNC state, bit differences between x[i,:],y[i,:] symbols are accumulated in counters - accessible by get_cnt(), get_acc_cnt().
\n Metrics = sum(xor(X,Y)) is a number of bit differences between states of X,Y LSR (X[0:L-1], Y[0:L-1]).
\n Metrics, together with thresholds [T0,T1,T2] is used in \ref BERT_FSM state transitions.
\n Instant change of state is possible only in SYNC, nSYNC states.
\n In RELOAD state BERT acquires number of bits greater of equal with the length internal PRBS generator and after transits to PRESET state.
\n In transition from RELOAD to PRESET state, data shifted into X synchronize PRBS to the input bit stream by setting the state of PRBS using acquired input bits.
\n PRESET requires acquisition of L bits before making a transition.
\n In PRESET state data both shifted into X,Y - yet Y symbols are generated by PRBS
\n Bit errors are updated only in SYNC state
\n \ref BERT_FSM can transition to SYNC state if (metrics < T0) but exits SYNC state if (metrics > T1)
\n For operation BERT counter requires setting:
\n W - symbol size for X,Y LSR and PRBS LFSR
\n L - length of X,Y LSR - X.lenght >= PRBS.lenght due to PRBS X pre-setting in RELOAD state
\n M - PRBS generator index or LFSR taps [bvec]
\n [T0,T1,T2] - thresholds for state transitions
\n BER is to be calculated as BER=(err_cnt)/(bit_cnt)
\n There are two error counters 
\n - instant values with auto reset after readout with get_cnt() 
\n - permanent values accessible via get_acc_cnt()
\n BER= (T0/L) is estimation of maximum BER value measurable by BERT, since BER>(T1/L) causes transition to nSYNC state
 */
class bert
{
private:
  bvec one;               //!< single ce tick

  bert_fsm_state FSM;     //!<  to nSYNC bit_neq > T1
  bert_fsm_state y0;			//!< initial output y0, output for ce=0
  int metrics;


  lsr lsrx;               //!< input shift register
  lsr lsry;				        //!< prbs shift register
  lfsr prbs;              //!< prbs generator

  int cnt_bit;            //!< reset when read or by RESET or reset_counters command
  int cnt_err;            //!< reset when read or by RESET or reset_counters command

  int acc_cnt_bit;        //!< reset by RESET or reset_counters command
  int acc_cnt_err;        //!< reset by RESET or reset_counters command

  int bit_reloaded;	      //!< bit counter acquired in RELOAD state
  int bit_preset;		      //!< bit counter acquired in PRESET state

  int T0;                 //<! thresholds for entering SYNC
  int T1;                 //<! thresholds for leaving SYNC
  int T2;	                //<! thresholds for entering RELOAD
  int symbol_size;

  bert_adr adr;

public:

  bert()
  {
    FSM = BERT_FSM_RESET;
    cnt_bit = 0;  // only static const data members can be initialized within a class
    cnt_err = 0;
    acc_cnt_bit = 0;
    acc_cnt_err = 0;
    bit_reloaded = 0;
    bit_preset = 0;
    one = bvec("1");
  };

  ~bert()
  {
  };

  /*! set predefined bpoly for PRBS defined by PRBS sequence 2^M-1 -see SIM::lfsr.set_prbs()
  \param [in] M - [int] power of PRBS sequence size = 2^M-1
  */
  void set_prbs(int M);

  /*! set predefined bpoly for PRBS - see SIM::lfsr.set_poly()
  \param [in] bpoly - [bvec] feedback poly
  */
  void set_poly(const bvec &bpoly);

  /*! set symbol_size
  \param [in] W - [int] number of bits in a symbol
  */
  void set_symbol_size(int W);

  /*! set length of shift X,Y register  - see SIM::lsr.set_length()
  \note X.length >= PRBS.lenght due to PRBS X pre-setting in RELOAD state
  \param [in] L - [int] size of shift register
  */
  void set_length(int L);

  /*! set thresholds vector T=[T0,T1,T2] defining a state transition for calculated metrics
  \note see \ref BERT_FSM for state definition
  \note T0>T1>T2; T0/N is max BER to be measured
  \param [in] T - [ivec] = [T0,T1,T2]
  */
  void set_threshold(const ivec &T);

  /*! sets FSM state
  \note default state is BERT_RESET
  \param [in] state - [int] valid state is one of a \ref BERT_FSM states
  */
  void set_fsm(bert_fsm_state state);

  /*! sets initial/idle output state
  \param [in] yout - [bert_state] y0 = yout, must be \ref BERT_FSM states
  */
  void set_output(bert_fsm_state yout);

  /*! sets indirect register target for get_state()
  \param [in] a - [bert_adr] must be \ref BERT_ADDR address
  */
  void set_adr(bert_adr a);

  /*! get length of shift registers X,Y
  \return - [int] length
  */
  int get_length(void);

  /*! get size of input X,Y symbol
  \return - [int] length of the symbol
  */
  int get_symbol_size(void);

  /*! Get metrics thresholds
  \return - [ivec] [T0,T1,T2]
  */
  ivec get_threshold(void);

  /*! Get counters
  \return - [ivec] [cnt_err, cnt_bit]
  */
  ivec get_cnt(void);

  /*! Get accumulated counters
  \return - [ivec] [acc_cnt_err, acc_cnt_bit]
  */
  ivec get_acc_cnt(void);

  /*! Get current value of metrics
  \return - [int] sum(xor(X[],Y[]))
  */
  int get_metrics();

  /*! Get current FSM state
  \return - [bert_state] \ref BERT_FSM
  */
  bert_fsm_state get_fsm(void); // y0 = FSM

  /*! get state of shift register addressed by last bert.set_adr() command
  \return - [bvec] state of shift X,Y,PRBS register
  */
  bvec get_state();

  /*! get connection poly of internal PRBS generator
  \return - [bvec] poly_gen
  */
  bvec get_poly();

  /*! get initial/idle output state
  \return - [bvec] y0 is \ref BERT_FSM
  */
  bert_fsm_state get_output(void);

  /*! reset all counters
  */
  void clear_counters(void);

  /*!\ For active clock ticks ce[i] input symbol x[i,:] is compared with internally generated symbol y[i,:].
  \n x[i,:],y[i,:] are shifted into X,Y LSR, metrics is calculated , and using metrics thresholds \ref BERT_FSM state is set.
  \n Bit errors sum(xor(x,y)) are accumulated in error counters when BERT is in the SYNC state.
  \param ce - [bvec] clock_enable vector
  \param x -  [bmat] input binary symbol to be compared with internally generated y[i,:] by PRBS LFSR
  \return y - [ivec] internal state \ref BERT_FSM ;
  */
  ivec process(const bvec &ce, const bmat &x);

};


}
#endif //SIM_bert
