/*!
* \brief - SIM::cofdm_dem - COFDM demodulator - derived from itpp::OFDM
* \author maki
*/

#ifndef COFDM_DEM_HPP
#define COFDM_DEM_HPP

#include <itpp/itbase.h>
#include <itpp/itcomm.h>

#include <complex>
#include "sim\_sim_extension.hpp"

using namespace itpp;
using namespace std;


namespace SIM
{

/*!
 * \brief COFDM demodulator \n
 * Core immediate functions SIM::cofdm_dem.set_symbol(), SIM::cofdm_mod.get_carriers() \n
 * Core clocked functions SIM::cofdm_dem.process() \n
 * \details
 * itpp::OFDM derived class
 */
class cofdm_dem : public itpp::OFDM
{
private:
  int	 NFFT;			  //!< N=2^k - Nfft
  int  CP;          //!< CP - circular prefix
  double scale;     //!< scaling factor
  cvec symbol;      //!< y0 - complex symbols - modulated carriers  y0=[CP|INV_FFT(carriers)]

public:

  cvec y0;		      //!< y0 - demodulated carriers (frequency domain)
  cofdm_dem()
  {
    NFFT = 256;
    CP = 16;
    OFDM::set_parameters(NFFT, CP, 1);
    scale = std::sqrt(static_cast<double>(NFFT)) / std::sqrt(static_cast<double>(NFFT + CP));
    y0.set_size(NFFT);
    symbol.set_size(NFFT);
  };
  
  ~cofdm_dem() 
  { 
  };

  /*! set FFT size (N=2^k)
  \param [in] n - FFT size
  */
  void set_NFFT(int n);

  /*! set size of CP - Circular Prefix
  \param [in] cp -  CP size
  */
  void set_CP(int cp);

  /*! set scale - default scaling is set by set_NFFT() and set_CP()
  \param [in] s - scale
  */
  void set_scale(double s);

  /*! set symbol - demodulator input - complex time domain samples x = [CP|INV_FFT(CC)]] 
  \param [in] x - complex time domain samples 
  */
  void set_symbol(cvec x);

  /*! get FFT size
  \return - [int] FFT size
  */
  int get_NFFT();

  /*! get CP size
  \return - [int] CP size
  */
  int get_CP();

  /*! get scaling factor
  \return - [double] scaling factor
  */
  double get_scale();

  /*! get symbol - demodulator input - complex time domain samples x = [CP|INV_FFT(CC)]]
  \return [cvec] complex time domain samples
  */
  cvec get_symbol();

  /*! get carriers - by setting new demodulator output - complex carriers CC = [I+jQ] 
  \return - [cvec] complex carriers CC = [I+jQ]
  */
  cvec get_carriers();

  /*! for active clock ticks ce[i] and for cofdm symbol x[i,:] return QAM, BPSK carriers y[i,:].
  \param ce - bvec[:] - clock_enable vector,
  \param x -  cmat[:,NFFT+CP] - cofdm symbols matrix,
  \return y - cmat[:,NFFT] - carriers matrix - QAM for data and BPSK for pilot carriers.
  */
  cmat process(const bvec &ce, const cmat &x);

};


}
#endif //SIM_cofdm_dem
