/*xxx
* \brief - SIM::cofdm_demap - demap [I,Q] QAM data carriers into QAM codes and BPSK pilot carriers into booleans  
* \author maki
*/

#ifndef COFDM_DEMAP_HPP
#define COFDM_DEMAP_HPP

#include <complex>
#include <itpp/itbase.h>
#include <itpp/itcomm.h>
#include "sim_qam_dem.hpp"
#include "sim\_sim_extension.hpp"

using namespace itpp;
using namespace std;

namespace SIM
{

/*!
* \brief de-mapper of COFDM carriers into QAM and BPSK symbol codes \n
* Core immediate functions: SIM::cofdm_demap.set_carriers(), SIM::cofdm_demap.get_data(), SIM::cofdm_demap.get_pilots().\n
* Core clocked functions: SIM::cofdm_demap.process().\n
* \details de-maps QAM codes for data carriers and booleans for BPSK pilot carriers
*/
class cofdm_demap
{
private:
  cvec carriers;        //!< COFDM carriers set by set_carriers()

  int	 NFFT;			      //!< NFFT=2^k - Nfft
  double PA;            //!< BPSK pilot amplitude (pilot[i]? PA+j0:-PA+j0)

  qam_dem qamdem;       //!< qam demodulator
  ivec data_carriers;   //!< indexes of QAM carriers for data 
  ivec pilots_carriers; //!< indexes of BPSK carriers for pilots
  ivec zero_carriers;   //!< indexes of zero carriers

public:
  ivec y0;		          //!< y0[i] - QAM_CODES, BOOL_PILOTS for data carriers and pilots

  cofdm_demap()
  {
    NFFT = 256;
    PA = 0.5;
    qamdem.set_size(4);
    y0.set_size(NFFT);
    carriers.set_length(NFFT);
    carriers = to_cvec(zeros(NFFT), zeros(NFFT));
    data_carriers = to_ivec(itpp::linspace(0, NFFT - 1, NFFT));
    pilots_carriers.set_size(0);
    zero_carriers.set_size(0);
  };
  
  ~cofdm_demap() 
  { 
  };

  /*! set FFT size (2^k)
  \param [in] n - FFT size
  */
  void set_NFFT(int n);

  /*! set QAM size (4^k)
  \param [in] m - QAM size
  */
  void set_qam_size(int m);

  /*! set pilot amplitude
  \param [in] pa - pilot amplitude (pilot ? pa+j0:-pa+j0)-> y0[pilot_carrier(i)]
  */
  void set_PA(double pa);

  /*! set data carrier indexes
  \param [in] dc_indx - [ivec] data carrier indexes, 0 =< index < NFFT
  */
  void set_data_carriers(ivec dc_indx);
  
  /*! set pilot carrier indexes
  \param [in] pc_indx - [ivec] pilot indexes, 0 =< index < NFFT
  */
  void set_pilots_carriers(ivec pc_indx);
  
  /*! set zero carrier indexes
  \param [in] zc_indx - [ivec] zero carrier indexes, 0 =< index < NFFT
  */
  void set_zero_carriers(ivec zc_indx);

  /*! set carriers as QAM [I+jQ] for data carriers , BPSK [+/-PA+j0] pilots , zero [0+j0] set by set_data() and set_pilots()
  \param [in] c [cvec] COFDM symbol 
  */
  void set_carriers(cvec c);

  /*! get QAM codes y[i] = QAMDEM[symbol[data_carrier(i)] for symbol set by set_symbol()
  \return [ivec] data carrier symbols
  */
  ivec get_data();
  
  /*! get bool pilot y[i] = (Real(symbol[pilot_carrier(i)]>=0)?1:0) for symbol set by set_symbol()
  \return [ivec] data carrier symbols
  */
  bvec get_pilots();

  /*! for active clock ticks ce[i] and for QAM,BPSK carriers x[i,:] return [PILOT|DATA] codes [ivec].
  \param ce - bvec[:] - clock_enable vector,
  \param x -  imat[:,PILOT|DATA] - symbols,
  \return y - imat[:,NFFT] - [PILOT|DATA] codes.
  */
  imat process(const bvec &ce, const cmat &x);

};


}
#endif //SIM_cofdm_mod
