/*!
* \brief - SIM::cofdm_map - map data (QAM codes) into [I,Q] QAM carriers, boolean pilots into BPSK carriers
* \author maki
*/

#ifndef COFDM_MAP_HPP
#define COFDM_MAP_HPP

#include <itpp/itbase.h>
#include <itpp/itcomm.h>

#include <complex>

#include "sim_qam_mod.hpp"

using namespace itpp;
using namespace std;

namespace SIM
{

/*!
 * \brief mapper of QAM and BPSK symbol codes into COFDM carriers [I,Q] \n 
 * Core immediate functions: SIM::cofdm_map.set_data(), SIM::cofdm_map.set_pilots(), SIM::cofdm_map.get_carriers()  \n  
 * Core clocked functions: SIM::cofdm_map.process() \n
 * \details 
 * map QAM carrier [I+jQ] for QAM codes[0..M] for data and BPSK carrier [+/-p+j0] for boolean pilots
 */
class cofdm_map
{
private:
  int	 NFFT;			        //!< NFFT=2^k - Nfft
  double PA;              //!< BPSK pilot amplitude (pilot[i]? PA+j0:-PA+j0)
  
  qam_mod qammod;         //!< qam modulator
  ivec data_carriers;     //!< indexes of QAM carriers for data 
  ivec pilots_carriers;   //!< indexes of BPSK carriers for pilots
  ivec zero_carriers;     //!< indexes of zero carriers

public:
  cvec y0;		            //!< y0[i] - QAM[i,q] PILOTS[i,q] for carriers and pilots

  cofdm_map()
  {
    NFFT = 256;
    PA = 0.5;
    qammod.set_size(4); 
    y0.set_size(NFFT);
    data_carriers = to_ivec(itpp::linspace(0, NFFT - 1, NFFT));
    pilots_carriers.set_size(0);
    zero_carriers.set_size(0);
  };
  
  ~cofdm_map() 
  { 
  };

  /*! set FFT size (2^k)
  \param [in] n - FFT size
  */
  void set_NFFT(int n);
  
  /*! set QAM size (4^k)
  \param [in] m - QAM size
  */
  void set_qam_size(int m);
 
  /*! set pilot amplitude
  \param [in] pa - pilot amplitude (pilot ? pa+j0:-pa+j0)-> y0[pilot_carrier(i)]
  */
  void set_PA(double pa);

  /*! set data carrier indexes 
  \param [in] dc_indx - [ivec] data carrier indexes, 0 =< index < NFFT  
  */
  void set_data_carriers(ivec dc_indx);

  /*! set pilot carrier indexes
  \param [in] pc_indx - [ivec] pilot indexes, 0 =< index < NFFT
  */
  void set_pilots_carriers(ivec pc_indx);

  /*! set zero carrier indexes
  \param [in] zc_indx - [ivec] zero carrier indexes, 0 =< index < NFFT
  */
  void set_zero_carriers(ivec zc_indx);

  /*! set QAM data carrier symbols, y0[data_carrier(i)] = QAM(x[i]), zero other carriers 
  \param [in] x - [ivec] data carrier symbols
  */
  void set_data(ivec x);

  /*! set pilots and zero carrier symbols, y0[pilot_carrier(i)] = (x[i]?pa+j0:-pa+j0) \n
  /* zero carriers are set together with pilots - y0[zero_carrier(k)] = 0.0 + j0.0
  \param [in] x - [ivec] pilot carrier boolean values
  */
  void set_pilots(bvec x);

  /*! get COFDM carriers - QAM data carriers [I+jQ], pilots [+/-PA+j0], zero [0+j0] set by set_data() and set_pilots()
  \return - [cvec] y0
  */
  cvec get_carriers();

  /*! for active clock ticks ce[i] and for [PILOT|DATA] codes x[i,:] return QAM, BPSK carriers y[i,:].
  \param ce - bvec[:] - clock enable vector,
  \param x -  imat[:,PILOT|DATA] - symbol codes,
  \return y - cmat[:,NFFT] - carriers - QAM for data and BPSK for pilot carriers.
  */
  cmat process(const bvec &ce, const imat &x);

};


}
#endif //SIM_cofdm_mod
