/*xxx
* \brief - SIM::cofdm_mod - COFDM modulator derived from itpp::OFDM
* \author maki
*/

#ifndef COFDM_MOD_HPP
#define COFDM_MOD_HPP

#include <itpp/itbase.h>
#include <itpp/itcomm.h>

#include <complex>
#include "sim\_sim_extension.hpp"


using namespace itpp;
using namespace std;

namespace SIM
{

/*!
 * \brief COFDM modulator \n
 * Core immediate functions SIM::cofdm_mod.set_carriers(), SIM::cofdm_mod.get_symbol() \n
 * Core clocked functions SIM::cofdm_mod.process() \n
 */
class cofdm_mod : public itpp::OFDM
{
private:
  int	 NFFT;			  //!< N=2^k - Nfft
  int  CP;          //!< CP - circular prefix
  double scale;     //!< scaling factor
  cvec carriers;    //!< complex QAM|BPSK carriers

public:
  cvec y0;		      //!< y0 - complex symbols - modulated carriers  y0=[CP|INV_FFT(carriers)]

  cofdm_mod()
  {
    NFFT = 256;
    CP = 16;
    OFDM::set_parameters(NFFT, CP, 1);
    scale = std::sqrt(static_cast<double>(NFFT + CP)) / std::sqrt(static_cast<double>(NFFT));
    y0.set_size(NFFT+CP);
    carriers.set_size(NFFT);
  };

  ~cofdm_mod() 
  { 
  };

  /*! set FFT size (N=2^k)
  \param [in] n - FFT size
  */
  void set_NFFT(int n);

  /*! set size of CP - Circular Prefix  
  \param [in] cp -  CP size
  */
  void set_CP(int cp);

  /*! set scale - default scaling is set by set_NFFT() and set_CP()
  \param [in] s - scale
  */
  void set_scale(double s);

  /*! set carriers - modulator input QAM|BPSK [I+jQ]  
  \param [in] x - [cvec] complex carriers
  */
  void set_carriers(cvec x);

  /*! get FFT size
  \return - [int] FFT size
  */
  int get_NFFT();

  /*! get CP size
  \return - [int] CP size
  */
  int get_CP();

  /*! get scaling factor
  \return - [double] scaling factor
  */
  double get_scale();

  /*! get carriers - modulator input QAM|BPSK [I+jQ] 
  \return - [cvec] complex carriers
  */
  cvec get_carriers();

  /*! get symbol - by setting new modulator output - complex time domain symbol y0=[CP|INV_FFT(carriers)]
  \return - [cvec] y0
  */
  cvec get_symbol();

  /*! for active clock ticks ce[i] and for QAM,BPSK carriers x[i,:] return time domain cofdm symbol y[i,:].   
  \param ce - bvec[:] - clock_enable vector,
  \param x -  cmat[:,NFFT] - QAM, BPSK carriers matrix,
  \return y - cmat[:,NFFT+CP] -  cofdm symbols matrix.
  */
  cmat process(const bvec &ce, const cmat &x);

};


}
#endif //SIM::cofdm_mod
