/*xxx
* \brief - SIM::lfsr for linear feedback shift register
* \author maki
*/



#ifndef SIM_LFSR_HPP
#define SIM_LFSR_HPP

#include <itpp/itbase.h>
#include <itpp/comm/sequence.h>

using namespace itpp;
using namespace std;


namespace SIM
{

/*!
 * \brief linear feedback shift register \n
 * Core function: SIM::lfsr.generate() \n
 * \details itpp::LFSR derived class with clocked interface
 * \note itpp::LFSR is in the Fibonacci form
 * \n Example: 1+X3+X4  for PRBS  sequence  = 2^4-1
 * \n set_prbs(4) -> set_poly with a bpoly=bvec("1 0 0 1 1");
 * <pre>
 *       1 +     X3+X4
 *       0  1  2  3  4
 *      [1  0  0  1  1]
 * </pre>
 * \n By definition there is no X^0 cell - first cell - state[0] is represented by X^1.
 * \n Threfore for LFSR of lenght N, state is bvec with lenght of (N-1)
 * \n The implementation is like J.83B standard p.9 -> PRBS randomizer
 * \n For one active tick ce[i] a single symbol represented as bvec with symbol_size = W bits is created:
 * \n y[i,:] = [x0(W-1), x0(W-2),...,x0(1), x0(0)] (default: reversed = true - x0(0) is oldest bit as for a generation time)
 * \n where each x0 bit is calculated as:
 * \n { bin x0 = state[0:N-1] * bpoly[1:N]; state.shift_right(x0); return x0; }
 * \n Output y for ce[M] ticks is bmat with a size y = bmat[M,W]
 * \note rev_flag controls if output is reversed, default value of rev_flag is TRUE
 * \note when output is revesed symbols s[k+1] = [b[n+5],b[n+4],b[n+3]], s[k] = [b[n+2],b[n+1],b[n]]
 * \n can be easily be used in BERT synchronization [s[k+1]|s[k]], because time scale is continous
 * \n
  */

class lfsr : public itpp::LFSR
{
private:
  bvec y0;			    //!< registered output, not initialized by constructor
  bvec s0;			    //!< reset state
  int symbol_size;	    //!< symbol size is size of row in output bmat
  bvec poly_gen;          //!< connection poly set by PRBS or set_poly()
  bool rev_flag;          //!< when set output and state are reversed

public:
  lfsr()
  {
    rev_flag = true;   // by default it is not compatible with LFSR to provide easy sync with bit symbols
    symbol_size = 1;
  };
  ~lfsr()
  {

  };

  /*! set feedback poly [g0,g1,...,gr] =1+g1*D+g2*D^2+...+gr*D^r
  \n and set default value s0=[10000000...] for reset state and executes reset !
  \note  set_poly() or set_prbs() are minimal set-up for symbol_size == 1
  \note - internally LSB is leftmost, output might be reversed if rev_flag is true
  \param [in] bpoly - [bvec] feedback poly
   */
  void set_poly(const bvec &bpoly);

  /*! set predefined bpoly for PRBS defined by PRBS sequecne 2^M-1
  \n and execute set_poly() for selected bpoly
  \note  set_poly() or set_prbs()  and set_symbol_size() are minimal set-up
  \note - bpoly are defined for [3..9,15,23,31] - source wikipedia, xilinx xapp052.pdf
  \param [in] M - [int] power of PRBS sequence size = 2^M-1
   */
  void set_prbs(int M);

  /*! set symbol_size
  \note  set_poly() or set_prbs()  and set_symbol_size() are minimal set-up
  \note - for one ce tick LFSR is shifted W times to create on symbol
  \param [in] W - [int] number of LFSR shifts and bits in output bmat for one ce tick
   */
  void set_symbol_size(int W);

  /*! sets value of internal forced by reset s0
  \note - only sets variable, does not executes RESET
  \param [in] binit - [bvec] initial internal state after RESET
   */
  void set_reset_state(const bvec &binit);

  /*! sets initial/idle output vector
  \note yout is a bvec of symbol size (W) length
  \param [in] yout - [bvec] y0 = yout
  */
  void set_output(const bvec &yout);

  /*! sets value of rev_flag
  \note when rev_flag is true, output is bit reversed, default value is true,
  \param [in] flag - [bvec] rev_flag = flag
  */
  void set_rev_flag(bool flag);

  /*! get connection poly
  \return - [bvec] y0 = yout
  */
  bvec get_poly();

  /*! get state binary vector
  \note state is reversed when rev_flag is set
  \return - [bvec] state
  */
  bvec get_state();

  /*! get symbol size
  \return - [int] W
  */
  int get_symbol_size();

  /*! get reset state binary vector
  \return - [bvec] s0
  */
  bvec get_reset_state();

  /*! get initial/idle output vector
  \return - [bvec] y0 ,sizeof(y0)=W
  */
  bvec get_output();

  /*! get value of rev_flag
  \note when rev_flag is true, output is bit reversed, default value is true,
  \return - [bool] rev_flag
  */
  bool get_rev_flag();


  /*! for active clock ticks [ce] generate one symbol (W bits) in output bmat y
  * \param ce - [bvec] - clock enable vector - sizeof(ce)=N
  * \return y - [bmat] - output symbols matrix - sizeof(y)=[N,W]
  */
  bmat generate(const bvec &ce);

};


}
#endif //SIM_lfsr
