/*xxx
* \brief - SIM::lsr - SIM layer implementation of linear shift register
* \author maki
*/



#ifndef SIM_LSR_HPP
#define SIM_LSR_HPP

#include <itpp/itbase.h>


using namespace itpp;
using namespace std;

namespace SIM
{

/*!
  \brief linear shift register \n
  Core function: SIM::lsr.process() \n
  \details linear shift register with clocked interface as used in BERT
  \n Shift register (aka memory) is a bvec with size of L bits
  \n Input is reprsented by bmat[N,W] for N ce[] ticks
  \n For i-th tick one row in x bmat represents one symbol with W bits
  \n x[i,:] = [x0(0), x0(1),...x0(W-1)]
  \n For active tick ce[i] Input symbol is shifted in memory as W - LSB bits (from index 0)
  \n mem(i) = [x[0:W-1] | mem[0:L-W-1]]
  \n output is bvec with W bit shifted out
  \n y[i,:] = m[L-W:L-1]
  \n
*/

class lsr
{
private:
  bvec y0;			//!< inital output y0, output for ce=0
  bvec s0;			//!< reset vector
  int symbol_size;	//!< W - size(x[i,:]) = size(y[i,:]) input/output symbol_size
  bvec memory;
  bvec shift(const bvec &bvin);

public:
  lsr()
  {

  };

  ~lsr()
  {

  };

  /*! set lenght of shift register
  \note also sets default reset bvec s0 = zeros(L) and resets register
  \note  set_length() and set_symbol_size() are minimal set-up
  \param [in] L - [int] size of shift register (aka memory)
  */
  void set_length(int L);

  /*! set size of a symbol
  \note W it checked to be smaller or equal with lenght
  \note also sets default initial/idle bvec y0 = zeros(W)
  \note  set_length() and set_symbol_size() are necessary minimal set-up
  \param [in] W - [int] size of a symbol
  */
  void set_symbol_size(int W);

  /*! set reset vector
  \note can be redefined after setting length of shift register
  \param [in] binit - [bvec] reset vector
  */
  void set_reset_state(const bvec &binit);

  /*! sets initial/idle output vector
  \note yout is a bvec of symbol size (W) length, initialized by set_symbol_size
  \param [in] yout - [bvec] y0 = yout
  */
  void set_output(const bvec &yout);

  /*! set state of shift (aka memory) register
  \param [in] state - [bvec] vector to set
  */
  void set_state(const bvec &state);

  /*! get state of shift register (aka memory)
  \return - [bvec] state of shift register
  */
  bvec get_state(void);

  /*! get reset binary vector
  \return - [bvec] reset vector
  */
  bvec get_reset_state();

  /*! get initial/idle output vector
  \return - [bvec] initial vector y0
  */
  bvec get_output();

  /*! get length of shift register
  \return - [int] lenght
  */
  int get_length(void);

  /*! get size of input/output symbol
  \return - [int] lenght of symbol
  */
  int get_symbol_size();

  /*! for active clock ticks ce[i]
  \n right shift in one input symbol x[i,:] and shift out one symbol y[i,:]
  \param  ce - [bvec] - clock enable vector - sizeof(ce)=N
  \param  x - [bmat] - input symbols matrix - sizeof(x)=[N,W]
  \return y -  [bmat] - output symbols matrix - sizeof(y)=[N,W]
  */
  bmat process(const bvec &ce, const bmat &x);

};

}
#endif //SIM_lsr
