/*xxx
* \brief nco - NCO with fixed point Q1.N accumulator
* \author maki
*/


#ifndef SIM_NCO_HPP
#define SIM_NCO_HPP

#include <itpp/itbase.h>
#include "sim\_sim_extension.hpp"


using namespace itpp;
using namespace std;

namespace SIM
{

/*!
* \brief fixed point NCO - numerically controlled oscillator \n
* Core function: SIM::nco.process() \n
* \details
* Accumulated phase is in Q1.N format, operational type is Q(64-N).N \n
* input vector x [vec] is frequency control word \n
* fcw is saturated at (-1..+1) and converted from fractional double to Q1.N fixed point \n
* nco output is a matrix mat=[cy|ph] where:\n
* cy is overflow flag: +1 for positive -1 for negative \n
* ph is accumulated phase converted to double fractional format \n
*/
class nco
{
  int Nq_bits;
  q63_t q_scale;
  q63_t acc;
  vec y0;

public:

  nco()
  {
    y0.set_length(2);
    y0(0) = 0.0;
    y0(1) = 0.0;
    Nq_bits = 31;
    q_scale = (q63_t)pow2(Nq_bits); // 2^31
    acc = 0;
  };

  ~nco()
  {
  };

  /*! set N - number of bits for fractional representation in Q1.N
  \param [in] n - number of bits
  */
  void set_N(int n);

  /*! set state of phase accumulator
  \param [in] a - state of phase accumulator in double fractional format
  */
  void set_acc(double a);

  /*! set initial output value y0
  \param [in] yout - [vec] = [cy|ph]
  */
  void set_output(const vec &yout);

  /*! get N - number of bits for fractional representation in Q1.N  */
  int get_N();

  /*! get state of phase accumulator converted to double fractional format */
  double get_acc();

  /*! get initial output value y0 */
  vec get_output();

  /*! process input samples - x[i] for active clock ticks ce[i]
  * \param ce - bvec - clock enable vector
  * \param x -  vec  - input vector - fcw saturated at (-1..+1) in double fractional format
  * \return y - mat  - output matrix [cy|ph] cy =(1,0,-1), ph - accumulated phase in double fractional format
  */
  mat process(const bvec &ce, const vec &x);
};

}	   // namespace SIM
#endif //SIM_NCO_HPP

