/*xxx
* \brief - SIM::src  - sample rate converter with sinc fir reconstruction filter
* \author maki
*/


#ifndef SIM_SRC_HPP
#define SIM_SRC_HPP

#include <itpp/itbase.h>
#include "sim\_sim_enum_def.h"

using namespace itpp;
using namespace std;

namespace SIM
{
/*!
 * \brief sample rate converter with sinc reconstruction filter \n
 * Core function: SIM::src.process() \n
 */
class src
{
private:
  int  NTaps;               //!< size of fir sinc kernel
  win_type WIN;             //!< window type

  double *p_x_cb;           //!< input circular buffer pointer
  int x_cb_i;               //!< current sample circular buffer index

  /*! sinc y=sin(pi*x)/(pi*x)
  \param [in] x - - [double] function argument
  \return - [double] function value
  */
  double sinc(double x);    

  /*! impulse response h=h(nt), nt - normalized time nt=t/Ts  
  \param [in] nt - [double] - time in samples nt=(N+tau), tau - is a Ts fraction
  \return - [double] impulse response value
  */
  double h(double nt);    

public:

  double y0;                  //!< registered filter output

  /*! reconstructed kernel response h=h_t(tau), tau - normalized time tau=dT/Ts
  \param [in] tau - [double] - tau - is a Ts fraction
  \return - [double] impulse response value
  */
  double re_samp(double tau);
  
  /*! samples circular buffer
  \param [in] x_new  - [double] new sample
  */
  void xcb(double x_new);
  
  /*! default SRC uses 15 taps sinc kernel with hanning window
  */
  src()
  {
    y0 = 0.0;
    NTaps = 15; 							          // default kernel size
    p_x_cb = new double[NTaps];			    // circular buffer initialization
    for(int i = 0; i < NTaps; i++) p_x_cb[i] = 0.0;
    x_cb_i = 0;
    WIN = WIN_HANNING;                  // default sinc windowing
  };

  ~src()
  {
    delete [] p_x_cb;
  };

  /*! set sinc kernel size
  \param [in] n  - [int] kernel size 
  */
  void set_size(int n);

  /*! set window type
  \param [in] w  - [int] window type WIN_HANNING=1, WIN_KAISER_BESSEL=2, WIN_FLAT_TOP=3
  */
  void set_win_type(int w);
  
  /*! set output value y0
  \param [in] yout - [double] => y0
  */
  void set_output(double yout);

  /*! get sinc kernel size
  \return  -  [int] sinc fir kernel size 
  */
  int get_size(void);

  /*! get sinc window type
  \return  -  [int] sinc fir kernel window type WIN_HANNING=1, WIN_KAISER_BESSEL=2, WIN_FLAT_TOP=3
  */
  int get_win_type(void);

  /*! get output
  \return  -  [double] y0
  */
  double get_output(void);

  /*! process input samples xt[0,i] for active input clock ticks ce[0,i] \n
      for active output clock tick ce[1,i] reconstruct output y[i] for a given fractional time xt[1,i] \n   
  * \param ceio - bmat - input/output [cei|ceo] clock enable matrix
  * \param xt -  mat  - input sample and fractional time xt=[x|t] matrix
  * \return y - vec  - output sample vector 
  */
  vec process(const bmat &ceio, const mat &xt);
};

}	   // namespace SIM
#endif //SIM_SRC_HPP

