/*xxx
//
* SPUC - Signal processing using C++ - A DSP library
// Copyright(c) 1993-1996 Tony Kirke
// author="Tony Kirke" *
// Copyright (c) 2014, Tony Kirke. License: MIT License (http://www.opensource.org/licenses/mit-license.php)
* \brief - SPUC::fir  - (templated) class - from SPUC - with minor modifications
*/

#ifndef SPUC_FIR_HPP
#define SPUC_FIR_HPP

#include <itpp/itbase.h>

#include "copyright.h"
#include "spuc.hpp"
#include "sim\_sim_exception.hpp"

namespace SPUC
{

/*!
 * \class SPUC::fir
 * \brief  Template Class for Modeling a Finite Impulse Response filter.
 *
 * \n Template works for double, long, complex, etc
 * \n Taps initialized to zeros.
 */
template <class Numeric> class fir
{
public:
//! size of filter - public(?)
  int num_taps;
//! pointer to coeff created on the heap - public(?)
  Numeric* coeff;
//protected:
//! pointer to state created on the heap - public(?)
  Numeric* z;
//! output - public(?)
  Numeric output;

public:
//! Destructor
  ~fir(void)
  {
    if(num_taps > 0)
    {
      delete [] z;
      delete [] coeff;
    }
  };

  //! Constructor
  fir(void) : coeff(NULL), z(NULL)
  {
    ;
  };

  //! Constructor
  fir(int n) : coeff(NULL), z(NULL), num_taps(n)
  {
    int i;
    if(n > 0)
    {
      coeff = new Numeric[n];
      z = new Numeric[n];
      for(i = 0; i < n; i++) z[i] = coeff[i] = 0;
    }
    else
    {
      throw sim_exception("fir.fir size n<= 0", n);
    }
  };

  //! Set size of Filter
  void set_size(int n)
  {
    int i;
    num_taps = n;
    if(n > 0)
    {
      coeff = new Numeric[n];
      z = new Numeric[n];
      for(i = 0; i < n; i++) z[i] = coeff[i] = 0;
    }
    else
    {
      throw sim_exception("fir.set_size - size n<= 0", n);
    }

  };

  //! Set tap weights - single
  void set_taps(int i, Numeric tap)
  {
    if((i < num_taps) && (i >= 0))
    {
      coeff[i] = tap;
    }
    else
    {
      throw sim_exception("fir.set_taps - bad index ", i);
    }
  };

  //! Set tap weights - vector
  void set_taps(Vector<Numeric> v)
  {
    set_size(v.length());
    if(num_taps)
    {
      for(int i = 0; i < num_taps; i++) coeff[i] = v[i];
    }
    else
    {
      throw sim_exception("fir.set_taps - num_taps <= 0", num_taps);
    }

  };

  //! Reset
  void reset()
  {
    for(int i = 0; i < num_taps; i++) z[i] = 0;
    output = 0;
  };
  //! Get sum of coefficients
  Numeric coeff_sum()
  {
    int i;
    Numeric s;
    for(s = 0, i = 0; i < num_taps; i++) s += coeff[i];
    return(s);
  };

  //! Get fir size
  int get_size(void)
  {
    return(num_taps);
  };

  //! Get taps as vector
  Vector<Numeric> get_taps(void)
  {
    Vector<Numeric> V(num_taps);
    for(int i = 0; i < num_taps; i++) V[i] = coeff[i];
    return(V);
  };

  //! Get input as vector
  Vector<Numeric> get_input(void)
  {
    Vector<Numeric> V(num_taps);
    for(int i = 0; i < num_taps; i++) V[i] = z[i];
    return(V);
  };


  //! Get current output
  Numeric out()
  {
    return(output);
  };

  //! Clock in new sample & compute current output
  Numeric check(int i)
  {
    if((i < num_taps) && (i >= 0))
    {
      return(z[i]);
    }
    else
    {
      throw sim_exception("fir.check - bad index ", i);
    }

  };

  //! Update filter by inputting 1 sample and returning convolved output sample.
  Numeric clock(Numeric in)
  {
    return(update(in));
  };

  //! Update filter by inputting 1 sample and returning convolved output sample.
  Numeric update(Numeric in)
  {
    int i;
    // Update history of inputs
    for(i = num_taps - 1; i > 0; i--) z[i] = z[i - 1];
    // Add new input
    z[0] = in;
    // Perform FIR
    for(output = 0, i = 0; i < num_taps; i++) output += coeff[i] * z[i];
    return(output);
  };

}; // class fir

} // namespace SPUC
#endif
