/*xxx
* \brief - first layer interface to the sci_gateway
* Data are on scilab stack, functions are redirected by gateway
* \n Code has to be pure C due to the problems with mex.h in C++
* \author maki
*/

#ifdef __cplusplus
extern "C" {
#endif

#include "scilab\scilab_gate.h"
#include "Scierror.h"
#include "localization.h"
#include "sciprint.h"
#include "MALLOC.h"


#include "sci\_sci_if.h"
#include "sci\_sci_enum_def.h"
#include "sci\_sci_types.h"

// var struct is defined by project - in sub-folder scilab, scicoslab, scipy
#include "_sci_if_var_struct.h"

#include "__debug.h"


  /*
   * \brief C interface to scilab sci_create
   *
   * Example: p_fir1=sci_create(SCI_FIR);
   * input SCI_FIR = type of SCI_TYPE object
   * output p_fir1 = pointer to new instance of object
   */

  static int scilab_err = SCI_ERR_OK;
  static int scilab_info = 0;
  static char scilab_err_msg[SCI_ERR_MSG_LEN];

  void sci_if_err_set(int e)
  {
    scilab_err = e;
  }

  void sci_if_err_msg_set(const char *p_msg)
  {
    strcpy_s(scilab_err_msg, SCI_ERR_MSG_LEN, p_msg);
  }

  void sci_if_info_set(int i)
  {
    scilab_info = i;
  }

  /*
   * \brief C interface to sci_create
   *
   * Example: p_fir1 = sci_create(SCI_FIR);
   * input1 SCI_TYPE (standard enum type == double)
   * output pointer to created SCI_TYPE object
   */

  int scilab_create(char *fname, unsigned long fname_len)
  {
    SciErr sciErr;
    int internalErr = 0;
    int *piAddr = NULL;
    void *p_sci;


    debug_1("[scilab_create] ..................... \r\n");

    //check input and output arguments
    CheckInputArgument(pvApiCtx, 1, 1);
    CheckOutputArgument(pvApiCtx, 1, 1);

    //get variable address of the first input argument
    sciErr = getVarAddressFromPosition(pvApiCtx, 1, &piAddr);
    if(sciErr.iErr == 0)
    {
      if(isDoubleType(pvApiCtx, piAddr))
      {
        if(isScalar(pvApiCtx, piAddr))
        {
          if(!isVarComplex(pvApiCtx, piAddr))
          {
            double dblArg	= 0;
            internalErr = getScalarDouble(pvApiCtx, piAddr, &dblArg);
            if(internalErr == 0)
            {
              scilab_err = 0;
              p_sci = sci_if_create((int)dblArg);
              if(p_sci == NULL)
              {
                internalErr = API_ERROR_INVALID_POINTER;
                debug_1("[scilab_create] Error: p_sci is NULL \r\n");
              }
              else if(scilab_err)
              {
                debug_1("[scilab_create] Error by sci_if_create() ???\r\n");
                sciprint("\n%s:\n info:%d\n", scilab_err_msg, scilab_info);
                internalErr = scilab_err;
              }
              else
              {
                sciErr = createPointer(pvApiCtx, nbInputArgument(pvApiCtx) + 1, p_sci);
              }
            }
          }
          else
          {
            internalErr = API_ERROR_INVALID_COMPLEXITY;
            debug_1("[scilab_create] Error: arg1 is complex ??? \r\n");
          }
        }
        else
        {
          internalErr = API_ERROR_IS_VECTOR;
          debug_1("[scilab_create] Error: arg1 is a not a scalar ??? \r\n");
        }
      }
      else
      {
        internalErr = API_ERROR_INVALID_TYPE;
        debug_1("[scilab_create] Error: arg1 type is not double ??? \r\n");
      }
    }

    if(sciErr.iErr)
    {
      printError(&sciErr, 0);
      Scierror(999, "sciErr error has occurred: %d\n", sciErr.iErr);
      return sciErr.iErr;
    }

    if(internalErr)
    {
      Scierror(999, "internalErr error has occurred: %d\n", internalErr);
      return internalErr;
    }

    AssignOutputVariable(pvApiCtx, 1) = nbInputArgument(pvApiCtx) + 1;
    debug_1("p_sci = 0x%X \r\n", p_sci);
    debug_1("[scilab_create] +++++++++++++++++++++ \r\n");

    return 0;
  }

  /*
   * \brief C interface to sci_set
   *
   * Example: sci_set(p_fir1,SCI_TAPS,c0);
   * input1 p_fir1= pointer to the instance SCI_TYPE object
   * input2 SCI_TAPS = enum type - scilab default type - double
   * input3 c0 = variable - might be matrix (scalar) booleans, integers, doubles, complexes
   * output NONE
   */

  int scilab_set(char *fname, unsigned long fname_len)
  {
    SciErr sciErr;
    int internalErr = 0;

    int *piAddr1 = NULL;
    int *piAddr2 = NULL;
    int *piAddr3 = NULL;
    void *p_sci;               // arg1 - pointer
    double dblArg2	= 0;        // arg2 - enum - default scilab type
    // arg3 - variable scilab type
    int iType_x		= 0;
    int iRows_x		= 0;
    int iCols_x		= 0;
    int iComplex_x		= 0;
    double* pdblReal_x	= NULL;
    double* pdblImg_x	= NULL;
    int* piBool_x      = NULL;
    int* piData_x		= NULL;

    struct var_struct s_x;

    debug_3("[scilab_set] ..................... \r\n");
    //check input and output arguments
    CheckInputArgument(pvApiCtx, 3, 3);
    CheckOutputArgument(pvApiCtx, 0, 1); // 0,0 ???

    //get variable address of the first input argument
    sciErr = getVarAddressFromPosition(pvApiCtx, 1, &piAddr1);
    if(sciErr.iErr == 0)
    {
      if(isPointerType(pvApiCtx, piAddr1))
      {
        sciErr = getPointer(pvApiCtx, piAddr1, &p_sci); // you should't check if pointer is scalar
      }
      else
      {
        internalErr = API_ERROR_INVALID_TYPE;
        debug_1("[scilab_set] Error: arg1 type is not a pointer ??? \r\n");

      }
    }

    //get variable address of the second argument
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      sciErr = getVarAddressFromPosition(pvApiCtx, 2, &piAddr2);
      if(sciErr.iErr == 0)
      {
        if(isDoubleType(pvApiCtx, piAddr2))
        {
          if(isScalar(pvApiCtx, piAddr2))
          {
            if(!isVarComplex(pvApiCtx, piAddr2))
            {
              internalErr = getScalarDouble(pvApiCtx, piAddr2, &dblArg2);
            }
            else
            {
              internalErr = API_ERROR_INVALID_COMPLEXITY;
              debug_1("[scilab_set] Error: arg2 is complex ??? \r\n");
            }
          }
          else
          {
            internalErr = API_ERROR_IS_VECTOR;
            debug_1("[scilab_set] Error: arg2 is not a scalar ??? \r\n");
          }
        }
        else
        {
          internalErr = API_ERROR_INVALID_TYPE;
          debug_1("[scilab_set] Error: arg2 type is not double ???\r\n");
        }
      }
    }

    //get variable address of the third argument
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      sciErr = getVarAddressFromPosition(pvApiCtx, 3, &piAddr3);
      if(sciErr.iErr == 0)
      {
        //check type
        sciErr = getVarType(pvApiCtx, piAddr3, &iType_x);
        if(sciErr.iErr == 0)
        {
          switch(iType_x)
          {

            case sci_matrix: // "1 - A matrix of doubles\n"
              //get complexity
              iComplex_x	= isVarComplex(pvApiCtx, piAddr3);
              //check complexity
              if(iComplex_x)
              {
                //get size and data from Scilab memory
                sciErr = getComplexMatrixOfDouble(pvApiCtx, piAddr3, &iRows_x, &iCols_x, &pdblReal_x, &pdblImg_x);
                if(sciErr.iErr == 0)
                {
                  s_x.sci_type = SCI_TYPE_CMAT; // there are no scalars and vectors in scilab - matrix is native type
                  s_x.p_external.re_im.p_re = pdblReal_x;
                  s_x.p_external.re_im.p_im = pdblImg_x;
                  s_x.rows = iRows_x;
                  s_x.cols = iCols_x;
                  s_x.p_var = NULL;
                }
              }
              else
              {
                //get size and data from Scilab memory
                sciErr = getMatrixOfDouble(pvApiCtx, piAddr3, &iRows_x, &iCols_x, &pdblReal_x);
                if(sciErr.iErr == 0)
                {
                  s_x.sci_type = SCI_TYPE_MAT; // there are no scalars and vectors in scilab - matrix is native type
                  s_x.p_external.p_double = pdblReal_x;
                  s_x.rows = iRows_x;
                  s_x.cols = iCols_x;
                  s_x.p_var = NULL;
                }
              }
              break;

            case sci_boolean: // "4 - A matrix of booleans \n";
              sciErr = getMatrixOfBoolean(pvApiCtx, piAddr3, &iRows_x, &iCols_x, &piBool_x);
              if(sciErr.iErr == 0)
              {
                s_x.sci_type = SCI_TYPE_BMAT; // there are no scalars and vectors in scilab - matrix is native type
                s_x.p_external.p_bool = piBool_x;
                s_x.rows = iRows_x;
                s_x.cols = iCols_x;
                s_x.p_var = NULL;
              }
              break;

            case sci_ints:  // "8 - A matrix of integers\n - 32bit nothing else";
              sciErr = getMatrixOfInteger32(pvApiCtx, piAddr3, &iRows_x, &iCols_x, &piData_x);
              if(sciErr.iErr == 0)
              {
                s_x.sci_type = SCI_TYPE_IMAT; // there are no scalars and vectors in scilab - matrix is native type
                s_x.p_external.p_int = piData_x;
                s_x.rows = iRows_x;
                s_x.cols = iCols_x;
                s_x.p_var = NULL;
              }
              break;

            default:
              internalErr = API_ERROR_INVALID_TYPE;
              debug_1("[scilab_set] Error: arg3 type is not supported ???\r\n");
              break;
          }

        }
      }
    }

    // execute function
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      scilab_err = 0;
      sci_if_set(p_sci, (int)dblArg2, &s_x); //
      if(scilab_err)
      {
        debug_1("[scilab_set] Error by sci_if_get() ???\r\n");
        sciprint("\n%s:\n info:%d\n", scilab_err_msg, scilab_info);
        internalErr = scilab_err;
      }
    }

    if(sciErr.iErr)
    {
      printError(&sciErr, 0);
      Scierror(999, "sciErr error has occurred: %d\n", sciErr.iErr);
      return sciErr.iErr;
    }

    if(internalErr)
    {
      Scierror(999, "internalErr error has occurred: %d\n", internalErr);
      return internalErr;
    }

    debug_3("[scilab_set] +++++++++++++++++++++ \r\n");
    return 0;

  }

  /*
   * \brief C interface to sci_get
   *
   * Example: c0 = sci_get(p_fir1,SCI_TAPS);
   * input1 p_fir1 = pointer to the instance of SCI_TYPE object
   * input2 SCI_TAPS = enum type - scilab default type - double
   * output c0 = variable - might be matrix (scalar) of booleans, integers, doubles, complexes
   */

  int scilab_get(char *fname, unsigned long fname_len)
  {
    SciErr sciErr;
    int internalErr = 0;

    int *piAddr1 = NULL;
    int *piAddr2 = NULL;

    void *p_sci;               // arg1 - pointer
    double dblArg2	= 0;        // arg2 - enum - default scilab type
    // result - variable scilab type
    int iType_y		= 0;
    int iRows_y		= 0;
    int iCols_y		= 0;
    int iComplex_y		= 0;
    double* pdblReal_y	= NULL;
    double* pdblImg_y	= NULL;
    int* piBool_y      = NULL;
    int* piData_y		= NULL;

    struct var_struct s_y;


    debug_3("[scilab_get] ..................... \r\n");
    //check input and output arguments
    CheckInputArgument(pvApiCtx, 2, 2);
    CheckOutputArgument(pvApiCtx, 1, 1);

    //get variable address of the first input argument
    sciErr = getVarAddressFromPosition(pvApiCtx, 1, &piAddr1);
    if(sciErr.iErr == 0)
    {
      if(isPointerType(pvApiCtx, piAddr1))
      {
        sciErr = getPointer(pvApiCtx, piAddr1, &p_sci); // you should't check if pointer is scalar
      }
      else
      {
        internalErr = API_ERROR_INVALID_TYPE;
        debug_1("[scilab_get] Error: arg1 type is not a pointer ??? \r\n");
      }
    }

    //get variable address of the second argument
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      sciErr = getVarAddressFromPosition(pvApiCtx, 2, &piAddr2);
      if(sciErr.iErr == 0)
      {
        if(isDoubleType(pvApiCtx, piAddr2))
        {
          if(isScalar(pvApiCtx, piAddr2))
          {
            if(!isVarComplex(pvApiCtx, piAddr2))
            {
              internalErr = getScalarDouble(pvApiCtx, piAddr2, &dblArg2);
            }
            else
            {
              internalErr = API_ERROR_INVALID_COMPLEXITY;
              debug_1("[scilab_get] Error: arg2 is complex ??? \r\n");
            }
          }
          else
          {
            internalErr = API_ERROR_IS_VECTOR;
            debug_1("[scilab_get] Error: arg2 is not a scalar ??? \r\n");
          }
        }
        else
        {
          internalErr = API_ERROR_INVALID_TYPE;
          debug_1("[scilab_get] Error: arg2 type is not double ???\r\n");
        }
      }
    }

    //get variable address of the third argument
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      scilab_err = 0;
      s_y = sci_if_get(p_sci, (int)dblArg2);
      if(scilab_err)
      {
        debug_1("[scilab_get] Error by sci_if_get() ???\r\n");
        sciprint("\n%s:\n info:%d\n", scilab_err_msg, scilab_info);
        internalErr = scilab_err;
      }
    }

    // allocate memory, copy data, delete variable from c heap
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      // now when m,n are known - create [mxn] sci variable on stack
      iRows_y = s_y.rows;
      iCols_y = s_y.cols;
      //reserve space in scilab memory for a given type
      switch(s_y.sci_type)
      {
        case SCI_TYPE_COMPLEX:
        case SCI_TYPE_CVEC:
        case SCI_TYPE_CMAT:
          sciErr = allocComplexMatrixOfDouble(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &pdblReal_y, &pdblImg_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.re_im.p_re = pdblReal_y;
            s_y.p_external.re_im.p_im = pdblImg_y;
          }
          break;
        case SCI_TYPE_DOUBLE:
        case SCI_TYPE_VEC:
        case SCI_TYPE_MAT:
          sciErr = allocMatrixOfDouble(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &pdblReal_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.p_double = pdblReal_y;
          }
          break;
        case SCI_TYPE_BOOL:
        case SCI_TYPE_BVEC:
        case SCI_TYPE_BMAT:
          sciErr = allocMatrixOfBoolean(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &piBool_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.p_bool = piBool_y;
          }
          break;
        case SCI_TYPE_INT:
        case SCI_TYPE_IVEC:
        case SCI_TYPE_IMAT:
          sciErr = allocMatrixOfInteger32(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &piData_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.p_int = piData_y;
          }
          break;
        default:
          internalErr = API_ERROR_INVALID_TYPE;
          debug_1("[scilab_get] Error: result type is unknown ??? \r\n");
          break;
      }

    }

    if(sciErr.iErr)
    {
      printError(&sciErr, 0);
      Scierror(999, "sciErr error has occurred: %d\n", sciErr.iErr);
      return sciErr.iErr;
    }

    if(internalErr)
    {
      Scierror(999, "internalErr error has occurred: %d\n", internalErr);
      return internalErr;
    }

    // copy values from c-heap into scilab memory
    sci_if_copy_var(&s_y);
    // remove variable from c-heap
    sci_if_delete_var(&s_y);

    AssignOutputVariable(pvApiCtx, 1) = nbInputArgument(pvApiCtx) + 1;
    debug_3("[scilab_get] +++++++++++++++++++++ \r\n");
    return 0;
  }

  /*
   * \brief C interface to sci_exec
   *
   * Example: sci_exec(p_fir1,SCI_RESET);
   * input1 p_fir1 = pointer to the instance of SCI_TYPE object
   * input2 SCI_RESET = type of command
   * output NONE
   */

  int scilab_exec(char *fname, unsigned long fname_len)
  {
    SciErr sciErr;
    int internalErr = 0;

    int *piAddr1 = NULL;
    int *piAddr2 = NULL;

    void *p_sci;               // arg1 - pointer
    double dblArg2	= 0;        // arg2 - enum - default scilab type

    debug_3("[scilab_exec] ..................... \r\n");
    //check input and output arguments
    CheckInputArgument(pvApiCtx, 2, 2);
    CheckOutputArgument(pvApiCtx, 0, 1); // 0,0 ???

    //get variable address of the first input argument
    sciErr = getVarAddressFromPosition(pvApiCtx, 1, &piAddr1);
    if(sciErr.iErr == 0)
    {
      if(isPointerType(pvApiCtx, piAddr1))
      {
        sciErr = getPointer(pvApiCtx, piAddr1, &p_sci); // you should't check if pointer is scalar
      }
      else
      {
        internalErr = API_ERROR_INVALID_TYPE;
        debug_1("[scilab_exec] Error: arg1 type is not a pointer ??? \r\n");
      }
    }

    //get variable address of the second argument
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      sciErr = getVarAddressFromPosition(pvApiCtx, 2, &piAddr2);
      if(sciErr.iErr == 0)
      {
        if(isDoubleType(pvApiCtx, piAddr2))
        {
          if(isScalar(pvApiCtx, piAddr2))
          {
            if(!isVarComplex(pvApiCtx, piAddr2))
            {
              internalErr = getScalarDouble(pvApiCtx, piAddr2, &dblArg2);
            }
            else
            {
              internalErr = API_ERROR_INVALID_COMPLEXITY;
              debug_1("[scilab_exec] Error: arg2 is complex ??? \r\n");
            }
          }
          else
          {
            internalErr = API_ERROR_IS_VECTOR;
            debug_1("[scilab_exec] Error: arg2 is not a scalar ??? \r\n");
          }
        }
        else
        {
          internalErr = API_ERROR_INVALID_TYPE;
          debug_1("[scilab_exec] Error: arg2 type is not double ???\r\n");
        }
      }
    }

    // execute function
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      scilab_err = 0;
      sci_if_exec(p_sci, (int)dblArg2);
      if(scilab_err)
      {
        sciprint("\n%s:\n info:%d\n", scilab_err_msg, scilab_info);
        internalErr = scilab_err;
      }
    }

    if(sciErr.iErr)
    {
      printError(&sciErr, 0);
      Scierror(999, "sciErr error has occurred: %d\n", sciErr.iErr);
      return sciErr.iErr;
    }

    if(internalErr)
    {
      Scierror(999, "internalErr error has occurred: %d\n", internalErr);
      return internalErr;
    }

    debug_3("[scilab_exec] +++++++++++++++++++++ \r\n");
    return 0;
  }

  /*
   * \brief C interface to sci_gen
   *
   * Example: y = sci_gen(p_mod,ce);
   * input1 p_mod = pointer to the instance of SCI_TYPE object
   * input2 ce = boolean
   * output y  = variable of any type
   */

  int scilab_gen(char *fname, unsigned long fname_len)
  {
    SciErr sciErr;
    int internalErr = 0;

    int *piAddr1 = NULL;
    int *piAddr2 = NULL;
    void *p_sci;               // arg1 - pointer

    int iRows_ce		= 0;    // arg2 - ce boolean type
    int iCols_ce		= 0;
    int* piBool_ce     = NULL;

    int iType_y		= 0;    // result - variable scilab type
    int iRows_y		= 0;
    int iCols_y		= 0;
    int iComplex_y		= 0;
    double* pdblReal_y	= NULL;
    double* pdblImg_y	= NULL;
    int* piBool_y      = NULL;
    int* piData_y		= NULL;

    struct var_struct s_y;
    struct var_struct s_ce;


    debug_3("[scilab_gen] ..................... \r\n");
    //check input and output arguments
    CheckInputArgument(pvApiCtx, 2, 2);
    CheckOutputArgument(pvApiCtx, 1, 1);

    //get variable address of the first input argument
    sciErr = getVarAddressFromPosition(pvApiCtx, 1, &piAddr1);
    if(sciErr.iErr == 0)
    {
      if(isPointerType(pvApiCtx, piAddr1))
      {
        sciErr = getPointer(pvApiCtx, piAddr1, &p_sci); // you should't check if pointer is scalar
      }
      else
      {
        internalErr = API_ERROR_INVALID_TYPE;
        debug_1("[scilab_gen] Error: arg1 type is not a pointer ??? \r\n");
      }
    }

    sciErr = getVarAddressFromPosition(pvApiCtx, 2, &piAddr2);
    if(sciErr.iErr == 0)
    {
      if(isBooleanType(pvApiCtx, piAddr2))
      {
        sciErr = getMatrixOfBoolean(pvApiCtx, piAddr2, &iRows_ce, &iCols_ce, &piBool_ce);
        if(sciErr.iErr == 0)
        {
          s_ce.sci_type = SCI_TYPE_BMAT; // there are no scalars and vectors in scilab - matrix is native type
          s_ce.p_external.p_bool = piBool_ce;
          s_ce.rows = iRows_ce;
          s_ce.cols = iCols_ce;
          s_ce.p_var = NULL;
        }
      }
      else
      {
        internalErr = API_ERROR_IS_VECTOR;
        debug_1("[scilab_gen] Error: arg2 is not boolean ??? \r\n");
      }
    }

    scilab_err = 0;
    s_y = sci_if_gen(p_sci, &s_ce);

    if(scilab_err)
    {
      debug_1("[scilab_get] Error by sci_if_get() ???\r\n");
      sciprint("\n%s:\n info:%d\n", scilab_err_msg, scilab_info);
      internalErr = scilab_err;
    }

    // allocate memory, copy data, delete variable from c heap
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      // now when m,n are known - create [mxn] sci variable on stack
      iRows_y = s_y.rows;
      iCols_y = s_y.cols;
      //reserve space in scilab memory for a given type
      switch(s_y.sci_type)
      {
        case SCI_TYPE_COMPLEX:
        case SCI_TYPE_CVEC:
        case SCI_TYPE_CMAT:
          sciErr = allocComplexMatrixOfDouble(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &pdblReal_y, &pdblImg_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.re_im.p_re = pdblReal_y;
            s_y.p_external.re_im.p_im = pdblImg_y;
          }
          break;
        case SCI_TYPE_DOUBLE:
        case SCI_TYPE_VEC:
        case SCI_TYPE_MAT:
          sciErr = allocMatrixOfDouble(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &pdblReal_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.p_double = pdblReal_y;
          }
          break;
        case SCI_TYPE_BOOL:
        case SCI_TYPE_BVEC:
        case SCI_TYPE_BMAT:
          sciErr = allocMatrixOfBoolean(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &piBool_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.p_bool = piBool_y;
          }
          break;
        case SCI_TYPE_INT:
        case SCI_TYPE_IVEC:
        case SCI_TYPE_IMAT:
          sciErr = allocMatrixOfInteger32(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &piData_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.p_int = piData_y;
          }
          break;
        default:
          internalErr = API_ERROR_INVALID_TYPE;
          debug_1("[scilab_gen] Error: result type is unknown ??? \r\n");
          break;
      }
    }

    if(sciErr.iErr)
    {
      printError(&sciErr, 0);
      Scierror(999, "sciErr error has occurred: %d\n", sciErr.iErr);
      return sciErr.iErr;
    }

    if(internalErr)
    {
      Scierror(999, "internalErr error has occurred: %d\n", internalErr);
      return internalErr;
    }

    // copy values from c-heap into scilab memory
    sci_if_copy_var(&s_y);
    // remove variable from c-heap
    sci_if_delete_var(&s_y);

    AssignOutputVariable(pvApiCtx, 1) = nbInputArgument(pvApiCtx) + 1;

    debug_3("[scilab_gen] +++++++++++++++++++++ \r\n");
    return 0;
  }



  /*
   * \brief C interface to sci_proc
   *
   * Example: y = sci_proc(p_fir1,ce,x);
   * input1 p_fir1 = pointer to the instance of SCI_TYPE object
   * input2 ce - boolean
   * input2 x  - any variable
   * output y  - any variable
   */

  int scilab_proc(char *fname, unsigned long fname_len)
  {
    SciErr sciErr;
    int internalErr = 0;

    int *piAddr1 = NULL;
    int *piAddr2 = NULL;
    int *piAddr3 = NULL;
    void *p_sci;               // arg1 - pointer

    int iRows_ce		= 0;    // arg2 - ce boolean type
    int iCols_ce		= 0;
    int* piBool_ce     = NULL;

    int iType_x		= 0;    // arg3 - variable scilab type
    int iRows_x		= 0;
    int iCols_x		= 0;
    int iComplex_x		= 0;
    double* pdblReal_x	= NULL;
    double* pdblImg_x	= NULL;
    int* piBool_x      = NULL;
    int* piData_x		= NULL;


    int iType_y		= 0;    // result - variable scilab type
    int iRows_y		= 0;
    int iCols_y		= 0;
    int iComplex_y		= 0;
    double* pdblReal_y	= NULL;
    double* pdblImg_y	= NULL;
    int* piBool_y      = NULL;
    int* piData_y		= NULL;

    struct var_struct s_ce;
    struct var_struct s_x;
    struct var_struct s_y;

    debug_3("[scilab_proc] ..................... \r\n");

    //check input and output arguments
    CheckInputArgument(pvApiCtx, 3, 3);
    CheckOutputArgument(pvApiCtx, 1, 1);

    //get variable address of the first input argument
    sciErr = getVarAddressFromPosition(pvApiCtx, 1, &piAddr1);
    if(sciErr.iErr == 0)
    {
      if(isPointerType(pvApiCtx, piAddr1))
      {
        sciErr = getPointer(pvApiCtx, piAddr1, &p_sci); // you should't check if pointer is scalar
      }
      else
      {
        internalErr = API_ERROR_INVALID_TYPE;
        debug_1("[scilab_proc] Error: arg1 type is not a pointer ??? \r\n");
      }
    }

    sciErr = getVarAddressFromPosition(pvApiCtx, 2, &piAddr2);
    if(sciErr.iErr == 0)
    {
      if(isBooleanType(pvApiCtx, piAddr2))
      {
        sciErr = getMatrixOfBoolean(pvApiCtx, piAddr2, &iRows_ce, &iCols_ce, &piBool_ce);
        if(sciErr.iErr == 0)
        {
          s_ce.sci_type = SCI_TYPE_BMAT; // there are no scalars and vectors in scilab - matrix is native type
          s_ce.p_external.p_bool = piBool_ce;
          s_ce.rows = iRows_ce;
          s_ce.cols = iCols_ce;
          s_ce.p_var = NULL;
        }
      }
      else
      {
        internalErr = API_ERROR_IS_VECTOR;
        debug_1("[scilab_proc] Error: arg2 is not boolean ??? \r\n");
      }
    }

    //get variable address of the third argument
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      sciErr = getVarAddressFromPosition(pvApiCtx, 3, &piAddr3);
      if(sciErr.iErr == 0)
      {
        //check type
        sciErr = getVarType(pvApiCtx, piAddr3, &iType_x);
        if(sciErr.iErr == 0)
        {
          switch(iType_x)
          {

            case sci_matrix: // "1 - A matrix of doubles\n"
              //get complexity
              iComplex_x	= isVarComplex(pvApiCtx, piAddr3);
              //check complexity
              if(iComplex_x)
              {
                //get size and data from Scilab memory
                sciErr = getComplexMatrixOfDouble(pvApiCtx, piAddr3, &iRows_x, &iCols_x, &pdblReal_x, &pdblImg_x);
                if(sciErr.iErr == 0)
                {
                  s_x.sci_type = SCI_TYPE_CMAT; // there are no scalars and vectors in scilab - matrix is native type
                  s_x.p_external.re_im.p_re = pdblReal_x;
                  s_x.p_external.re_im.p_im = pdblImg_x;
                  s_x.rows = iRows_x;
                  s_x.cols = iCols_x;
                  s_x.p_var = NULL;
                }
              }
              else
              {
                //get size and data from Scilab memory
                sciErr = getMatrixOfDouble(pvApiCtx, piAddr3, &iRows_x, &iCols_x, &pdblReal_x);
                if(sciErr.iErr == 0)
                {
                  s_x.sci_type = SCI_TYPE_MAT; // there are no scalars and vectors in scilab - matrix is native type
                  s_x.p_external.p_double = pdblReal_x;
                  s_x.rows = iRows_x;
                  s_x.cols = iCols_x;
                  s_x.p_var = NULL;
                }
              }
              break;

            case sci_boolean: // "4 - A matrix of booleans \n";
              sciErr = getMatrixOfBoolean(pvApiCtx, piAddr3, &iRows_x, &iCols_x, &piBool_x);
              if(sciErr.iErr == 0)
              {
                s_x.sci_type = SCI_TYPE_BMAT; // there are no scalars and vectors in scilab - matrix is native type
                s_x.p_external.p_bool = piBool_x;
                s_x.rows = iRows_x;
                s_x.cols = iCols_x;
                s_x.p_var = NULL;
              }
              break;

            case sci_ints:  // "8 - A matrix of integers\n - 32bit nothing else";
              sciErr = getMatrixOfInteger32(pvApiCtx, piAddr3, &iRows_x, &iCols_x, &piData_x);
              if(sciErr.iErr == 0)
              {
                s_x.sci_type = SCI_TYPE_IMAT; // there are no scalars and vectors in scilab - matrix is native type
                s_x.p_external.p_int = piData_x;
                s_x.rows = iRows_x;
                s_x.cols = iCols_x;
                s_x.p_var = NULL;
              }
              break;

            default:
              internalErr = API_ERROR_INVALID_TYPE;
              debug_1("[scilab_set] Error: arg3 type is not supported ???\r\n");
              break;
          }

        }
      }
    }

    // execute function
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      scilab_err = 0;
      s_y = sci_if_proc(p_sci, &s_ce, &s_x);
      if(scilab_err)
      {
        debug_1("[scilab_proc] Error by sci_if_proc() ???\r\n");
        sciprint("\n%s:\n info:%d\n", scilab_err_msg, scilab_info);
        internalErr = scilab_err;
      }
    }

    // allocate memory, copy data, delete variable from c heap
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      iRows_y = s_y.rows;
      iCols_y = s_y.cols;
      //reserve space in scilab memory for a given type
      switch(s_y.sci_type)
      {
        case SCI_TYPE_COMPLEX:
        case SCI_TYPE_CVEC:
        case SCI_TYPE_CMAT:
          sciErr = allocComplexMatrixOfDouble(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &pdblReal_y, &pdblImg_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.re_im.p_re = pdblReal_y;
            s_y.p_external.re_im.p_im = pdblImg_y;
          }
          break;
        case SCI_TYPE_DOUBLE:
        case SCI_TYPE_VEC:
        case SCI_TYPE_MAT:
          sciErr = allocMatrixOfDouble(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &pdblReal_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.p_double = pdblReal_y;
          }
          break;
        case SCI_TYPE_BOOL:
        case SCI_TYPE_BVEC:
        case SCI_TYPE_BMAT:
          sciErr = allocMatrixOfBoolean(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &piBool_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.p_bool = piBool_y;
          }
          break;
        case SCI_TYPE_INT:
        case SCI_TYPE_IVEC:
        case SCI_TYPE_IMAT:
          sciErr = allocMatrixOfInteger32(pvApiCtx, nbInputArgument(pvApiCtx) + 1, iRows_y, iCols_y, &piData_y);
          if(sciErr.iErr == 0)
          {
            s_y.p_external.p_int = piData_y;
          }
          break;
        default:
          internalErr = API_ERROR_INVALID_TYPE;
          debug_1("[scilab_gen] Error: result type is unknown ??? \r\n");
          break;
      }
    }

    if(sciErr.iErr)
    {
      printError(&sciErr, 0);
      Scierror(999, "sciErr error has occurred: %d\n", sciErr.iErr);
      return sciErr.iErr;
    }

    if(internalErr)
    {
      Scierror(999, "internalErr error has occurred: %d\n", internalErr);
      return internalErr;
    }

    // copy values from c-heap into scilab memory
    sci_if_copy_var(&s_y);
    // remove variable from c-heap
    sci_if_delete_var(&s_y);

    AssignOutputVariable(pvApiCtx, 1) = nbInputArgument(pvApiCtx) + 1;

    debug_3("[scilab_proc] +++++++++++++++++++++ \r\n");
    return 0;

  }

  /*
   * \brief C interface to scilab sci_destroy
   *
   * Example: sci_destroy(p_fir1);
   * input SCI_FIR = type of SCI_TYPE object
   * output void
   */

  int scilab_destroy(char *fname, unsigned long fname_len)
  {
    SciErr sciErr;
    int internalErr = 0;

    int *piAddr1 = NULL;
    void *p_sci;               // arg1 - pointer

    debug_1("[scilab_destroy] ..................... \r\n");
    //check input and output arguments
    CheckInputArgument(pvApiCtx, 1, 1);
    CheckOutputArgument(pvApiCtx, 0, 1); // 0,0 ???

    //get variable address of the first input argument
    sciErr = getVarAddressFromPosition(pvApiCtx, 1, &piAddr1);
    if(sciErr.iErr == 0)
    {
      if(isPointerType(pvApiCtx, piAddr1))
      {
        sciErr = getPointer(pvApiCtx, piAddr1, &p_sci); // you should't check if pointer is scalar
      }
      else
      {
        internalErr = API_ERROR_INVALID_TYPE;
        debug_1("[scilab_destroy] Error: arg1 type is not a pointer ??? \r\n");

      }
    }
    // execute function
    if((sciErr.iErr == 0) && (internalErr == 0))
    {
      scilab_err = 0;
      debug_1("p_sci = 0x%X \r\n",  p_sci);
      sci_if_destroy(p_sci);
      if(scilab_err)
      {
        debug_1("[scilab_destroy] Error by sci_if_destroy() ???\r\n");
        sciprint("\n%s:\n info:%d\n", scilab_err_msg, scilab_info);
        internalErr = scilab_err;
      }
    }

    if(sciErr.iErr)
    {
      printError(&sciErr, 0);
      Scierror(999, "sciErr error has occurred: %d\n", sciErr.iErr);
      return sciErr.iErr;
    }

    if(internalErr)
    {
      Scierror(999, "internalErr error has occurred: %d\n", internalErr);
      return internalErr;
    }

    debug_1("[scilab_destroy] +++++++++++++++++++++ \r\n");
    return 0;
  }


#ifdef __cplusplus
}
#endif

