"""
 scipy object SCI_TYPE descriptor
"""

import numpy    as np
import ctypes   as ct
import copy     as copy
import sys, os, inspect

from scipy_def  import *
from scipy_dbg  import *

lib_folder = os.path.realpath(os.path.abspath(os.path.split(inspect.getfile( inspect.currentframe() ))[0]))
dll_file_path = os.path.join(lib_folder, 'scipy_itpp.dll')
dll_dbg_file_path = os.path.join(lib_folder, 'scipy_itpp_dbg.dll')
dll_dbg = 0
set_DBG(1)

#typedef void  (*py_error_handler_t)(char *p_msg, int arg1, int arg2);
def scipy_error_callback(error_msg, error_info, error_code):
    print error_msg , error_info, error_code
    sys.exit(-1)
    return


class exception_SCITYPE(Exception):
    def __str__(self): return 'exception_SCITYPE: Not a SCI_TYPE ...'

class exception_NDArrayDim(Exception):
    def __str__(self): return 'exception_NDArrayDim: Number of dims > 2 ...'

class exception_NDArrayNotCONTIGOUS(Exception):
    def __str__(self): return 'exception_NDArray: flags[C_CONTIGUOUS] is False ...'

class exception_NDArrayAttribute(Exception):
    def __str__(self): return 'exception_NDArrayAttribute: access raised exception ??? ...'


class ctypesComplexStruct(ct.Structure):
    _fields_ = [("real", ct.c_double), ("imag", ct.c_double)]

class sci_var_s(ct.Structure):
    """
    variable descriptor
    """
    _fields_ = [("p_pyVar", ct.c_void_p),\
                ("nRows", ct.c_int),\
                ("nCols", ct.c_int),\
                ("sciType", ct.c_int),\
                ("p_cVar", ct.c_void_p)\
                ]


def scalar_scipy_type(x):
    """
    return SCI_TYPE for a scalar argument
    """
    if isinstance( x, complex ):
        return SCI_TYPE_COMPLEX, ctypesComplexStruct(x.real, x.imag)    
    elif isinstance( x, float ):
        return SCI_TYPE_DOUBLE, ct.c_double(x)
    elif isinstance( x, bool ):
        return SCI_TYPE_BOOL , ct.c_bool(x)
    elif isinstance( x, int ):
        return SCI_TYPE_INT, ct.c_int(x)
    else :
        raise exception_SCITYPE
    return

def get_vector_sci_type(x):
    """
    return SCI_TYPE for a scalar argument
    """
    if issubclass(x.dtype.type, np.float):
        return SCI_TYPE_VEC
    elif issubclass(x.dtype.type, np.complex):
        return SCI_TYPE_CVEC    
    elif issubclass(x.dtype.type, np.integer):
        return SCI_TYPE_IVEC
    elif issubclass(x.dtype.type, np.bool_ ):
        return SCI_TYPE_BVEC
    else :
        raise exception_SCITYPE
    return

def get_matrix_sci_type(x):
    """
    return SCI_TYPE for a scalar argument
    """
    if issubclass(x.dtype.type, np.float):
        return SCI_TYPE_MAT
    elif issubclass(x.dtype.type, np.complex):
        return SCI_TYPE_CMAT    
    elif issubclass(x.dtype.type, np.integer):
        return SCI_TYPE_IMAT
    elif issubclass(x.dtype.type, np.bool_ ):
        return SCI_TYPE_BMAT
    else :
        raise exception_SCITYPE
    return

# this object is of dynamic type - set by py_allocation_callback
global_scipy_var = np.empty(1, dtype=np.float_)

#typedef void* (*py_allocator_t)(scipy_var_t scipy_var);
def scipy_allocation_callback(scipy_struct):
    """
    allocate memory for SCI_TYPE defined in scipy_struct  
    """
    
    global global_scipy_var

    DBG_2( 'py_allocation_callback scipy_struct.sciType = ', scipy_struct.sciType)
        
    if (scipy_struct.sciType == SCI_TYPE_MAT):
        global_scipy_var = np.empty([scipy_struct.nRows, scipy_struct.nCols], dtype=np.float_)
    elif (scipy_struct.sciType == SCI_TYPE_VEC) or (scipy_struct.sciType == SCI_TYPE_DOUBLE) :
        global_scipy_var = np.empty(scipy_struct.nRows * scipy_struct.nCols, dtype=np.float_)
    elif (scipy_struct.sciType == SCI_TYPE_CMAT):
        global_scipy_var = np.empty([scipy_struct.nRows, scipy_struct.nCols], dtype=np.complex_)
    elif (scipy_struct.sciType == SCI_TYPE_CVEC) or (scipy_struct.sciType == SCI_TYPE_COMPLEX) :
        global_scipy_var = np.empty(scipy_struct.nRows * scipy_struct.nCols, dtype=np.complex_)
    elif (scipy_struct.sciType == SCI_TYPE_IMAT):
        global_scipy_var = np.empty([scipy_struct.nRows, scipy_struct.nCols], dtype=np.int_)
    elif (scipy_struct.sciType == SCI_TYPE_IVEC) or (scipy_struct.sciType == SCI_TYPE_INT) :
        global_scipy_var = np.empty(scipy_struct.nRows * scipy_struct.nCols, dtype=np.int_)    
    elif (scipy_struct.sciType == SCI_TYPE_BMAT):
        global_scipy_var = np.empty([scipy_struct.nRows, scipy_struct.nCols], dtype=np.bool_)
    elif (scipy_struct.sciType == SCI_TYPE_BVEC) or (scipy_struct.sciType == SCI_TYPE_BOOL) :
        global_scipy_var = np.empty(scipy_struct.nRows * scipy_struct.nCols, dtype=np.bool_)    
    else :
        return ct.c_void_p(0).value                
    
    DBG_2( 'py_allocation_callback  p_py_var =', hex(global_scipy_var.ctypes.data))
    return global_scipy_var.ctypes.data_as(ct.c_void_p).value

def return_global_scipy_var():
    """
    for vector and arrays return copy of an ndarray object, for scalars return value (immutable)  
    """
    if global_scipy_var.ndim != 1 or global_scipy_var.size != 1 :
        return copy.deepcopy(global_scipy_var)
    else : 
        return global_scipy_var[0]
    
class scipy_var():
    
    def __init__(self,x):        
        """
        create class with variable descriptor as sci_var_s for 
        argument of SCI_TYPE_CLASS 
        """        
        if np.isscalar(x):            
            nRows = 1
            nCols = 1
            try : 
                sciType, self.ctype = scalar_scipy_type(x)
            except :
                exception_SCITYPE()
            p_py = ct.cast(ct.byref(self.ctype), ct.c_void_p) # it will work for object of ctype class
        
        else : # if not scalar - must be np.ndarray object
            try :
                if x.ndim == 1 and x.flags['C_CONTIGUOUS']: # vector
                    nRows = x.size
                    nCols = 1
                    try:
                        sciType = get_vector_sci_type(x)
                    except exception_SCITYPE:
                        exception_SCITYPE()
                    p_py = ct.cast(x.ctypes.data, ct.c_void_p) # cast to void * is required                                
                elif x.ndim == 2 and x.flags['C_CONTIGUOUS']: # matrix           
                    nRows = x.shape[0]
                    nCols = x.shape[1]
                    try:
                        sciType = get_matrix_sci_type(x)
                    except exception_SCITYPE:
                        exception_SCITYPE()
                    p_py = ct.cast(x.ctypes.data, ct.c_void_p) # cast to void * is required
                    
                else :  # something else than 2dim matrix - not supported...
                    if (x.ndim > 2) :
                        raise exception_NDArrayDim()
                    if not x.flags['C_CONTIGOUS']:
                        raise exception_NDArrayNotCONTIGOUS()
            except :
                exception_NDArrayAttribute()
        
        self.scipy_var_struct = sci_var_s(p_py, nRows, nCols, sciType, ct.c_void_p(0) )     
        return 

# factory functions are called with the result type as first argument, 
# and the callback functions expected argument types as the remaining arguments.
# Ctypes Cookbook - Dynamic allocation through callbacks
# This isn't the prettiest way to define the allocator, but there are a few bugs in ctypes that seem to make this the only way at present.
#typedef void* (*py_allocator_t)(scipy_var_t scipy_var);
ALLOCATOR = ct.CFUNCTYPE(ct.c_long, sci_var_s)
# Create the C  callback:
scipy_alloc_cb = ALLOCATOR(scipy_allocation_callback)

#typedef void  (*py_error_handler_t)(char *p_msg, int arg1, int arg2);
ERROR_HANDLER = ct.CFUNCTYPE(None, ct.c_char_p, ct.c_int, ct.c_int)
# Create the C callback:
scipy_error_cb = ERROR_HANDLER(scipy_error_callback)

# Load the dll library
if dll_dbg:
    _dll = np.ctypeslib.load_library(dll_dbg_file_path, '.')
else:    
    _dll = np.ctypeslib.load_library(dll_file_path, '.')

# << int scipy_init(py_allocator_t p_allocator, py_error_handler_t p_error_handler) >>
_dll.scipy_init.restype = None
_dll.scipy_init.argtypes = [ALLOCATOR, ERROR_HANDLER]

# << void * scipy_create(int sci_type) >>
_dll.scipy_create.restype = ct.c_void_p
_dll.scipy_create.argtypes = [ct.c_int]

# << void scipy_set(void *p_sci, int param, scipy_var_t *p_scipy_x) >>
_dll.scipy_set.restype = None
_dll.scipy_set.argtypes = [ct.c_void_p, ct.c_int, ct.POINTER(sci_var_s)]

# << void scipy_get(void *p_sci, int param) >>
_dll.scipy_get.restype = None
_dll.scipy_get.argtypes = [ct.c_void_p, ct.c_int]

# << void scipy_exec(void *p_sci, int param) >>
_dll.scipy_exec.restype = None
_dll.scipy_exec.argtypes = [ct.c_void_p, ct.c_int]

# << void scipy_gen(void *p_sci, scipy_var_t *p_scipy_ce) >>
_dll.scipy_gen.restype = None
_dll.scipy_gen.argtypes = [ct.c_void_p, ct.POINTER(sci_var_s)]

# << void scipy_proc(void *p_sci, scipy_var_t *p_scipy_ce, scipy_var_t *p_scipy_x) >>
_dll.scipy_proc.restype = None
_dll.scipy_proc.argtypes = [ct.c_void_p, ct.POINTER(sci_var_s), ct.POINTER(sci_var_s)]

# << void scipy_destroy(void *p_sci) >>
_dll.scipy_destroy.restype = None
_dll.scipy_destroy.argtypes = [ct.c_void_p]

#-------------------------------------

def SCI_CREATE(sci_type):
    p_sci = _dll.scipy_create(sci_type)
    return p_sci

def SCI_SET(p_sci, param, x):
    scipy_var_x = scipy_var(x)
    _dll.scipy_set(p_sci, param, ct.byref(scipy_var_x.scipy_var_struct))
    return

def SCI_GET(p_sci, param):
    _dll.scipy_get(p_sci, param)
    y = return_global_scipy_var()
    return y

def SCI_EXEC(p_sci, param):
    _dll.scipy_exec(p_sci, param)
    return

def SCI_GEN(p_sci, ce):
    scipy_var_ce = scipy_var(ce)
    _dll.scipy_gen(p_sci, ct.byref(scipy_var_ce.scipy_var_struct))
    y = return_global_scipy_var()
    return y

def SCI_PROC(p_sci, ce, x):
    scipy_var_ce = scipy_var(ce)
    scipy_var_x = scipy_var(x)
    _dll.scipy_proc(p_sci, ct.byref(scipy_var_ce.scipy_var_struct), ct.byref(scipy_var_x.scipy_var_struct))
    y = return_global_scipy_var()
    return y

def SCI_DESTROY(p_sci):
    _dll.scipy_destroy(p_sci)
    return

#-------------------------------------
# init library by definig callback
_dll.scipy_init(scipy_alloc_cb, scipy_error_cb)

if __name__ == '__main__':
# call all functions at least once    
    TAPS = 5
    N = 7
    c0 = np.ones(shape=(TAPS))*0.1
    clk = np.ndarray(shape=(2*N), dtype=bool)
    x = np.zeros(shape=(2*N))
    x[0] = 1.0
    clk[0:2*N] = True;

    p_cnt1 = SCI_CREATE(SCI_COUNTER)
    SCI_SET(p_cnt1, SCI_PARAM_N, 1)
    SCI_EXEC(p_cnt1, SCI_RESET)
    ce = SCI_GEN(p_cnt1, clk)
        
    p_fir1 = SCI_CREATE(SCI_FIR)            
    SCI_SET(p_fir1, SCI_TAPS, c0)
    SCI_EXEC(p_fir1, SCI_RESET)
    c1 = SCI_GET(p_fir1,SCI_TAPS)    
    x0 = SCI_GET(p_fir1, SCI_STATE)
    y = SCI_PROC(p_fir1, ce, x)
    
    print "clk =", clk.astype(int)
    print "ce  =", ce.astype(int)
    print "c0  =", c0
    print "c1  =", c1
    print "x   =", x
    print "y   =", y
    SCI_DESTROY(p_fir1);
    

