/*xxx
* \brief - SIM::bert - bit error rate tester
* \author maki
*/

#include "sim\sim_bert.hpp"
#include "sim\_sim_exception.hpp"
#include "__debug_level.h"
#include "debug.hpp"

namespace SIM
{

void bert::set_prbs(int M)
{
  prbs.set_prbs(M);
  return;
}

void bert::set_poly(const bvec &bpoly)
{
  prbs.set_poly(bpoly);
  return;
}

void bert::set_symbol_size(int W)
{
  lsrx.set_symbol_size(W);
  lsry.set_symbol_size(W);
  prbs.set_symbol_size(W);
  symbol_size = W;
  return;
}

void bert::set_length(int L)
{
  lsrx.set_length(L);
  lsry.set_length(L);
  return;
}

void bert::set_threshold(const ivec &T)
{
  if(T.length() == 3)
  {
    T0 = T(0);
    T1 = T(1);
    T2 = T(2);
  }
  else
  {
    throw sim_exception(" bert::set_threshold - length !=3 ", T.length());
  }
  return;
}

void bert::set_fsm(bert_fsm_state state)
{
  switch(state)
  {
    case BERT_FSM_RESET:
    case BERT_FSM_RELOAD:
    case BERT_FSM_PRESET:
    case BERT_FSM_nSYNC:
    case BERT_FSM_SYNC:
      FSM = state;
      break;

    default:
      throw sim_exception(" bert::set_fsm - unknown FSM state ", state);
  }
  return;
}

void bert::set_output(bert_fsm_state yout)
{
  switch(yout)
  {
    case BERT_FSM_RESET:
    case BERT_FSM_RELOAD:
    case BERT_FSM_PRESET:
    case BERT_FSM_nSYNC:
    case BERT_FSM_SYNC:
      y0 = yout;
      break;

    default:
      throw sim_exception(" bert::set_output - unknown state ", yout);
  }
  return;

}

void bert::set_adr(bert_adr a)
{
  switch(a)
  {
    case BERT_ADR_LSRX:
    case BERT_ADR_LSRY:
    case BERT_ADR_PRBS:
      adr = a;
      break;

    default:
      throw sim_exception(" bert::set_adr - unknown address ", a);
      break;
  }
  return;
}

int bert::get_length(void)
{
  return lsrx.get_length();
}

int bert::get_symbol_size(void)
{
  return symbol_size;
}


ivec bert::get_threshold(void)
{
  ivec T;

  T.set_size(3);
  T(0) = T0;
  T(1) = T1;
  T(2) = T2;
  return T;
}

ivec bert::get_cnt(void)
{
  ivec BER;

  BER.set_size(2);
  BER(0) = cnt_err;
  BER(1) = cnt_bit;
  cnt_bit = 0;
  cnt_err = 0;
  return BER;
}

ivec bert::get_acc_cnt(void)
{
  ivec B_CNT;

  B_CNT.set_size(2);
  B_CNT(0) = acc_cnt_err;
  B_CNT(1) = acc_cnt_bit;
  return B_CNT;
}


bvec bert::get_state()
{
  bvec bv;

  switch(adr)
  {
    case BERT_ADR_LSRX:
      bv = lsrx.get_state();
      break;

    case BERT_ADR_LSRY:
      bv = lsry.get_state();
      break;

    case BERT_ADR_PRBS:
      bv = prbs.get_state();
      break;

    default:
      throw sim_exception(" bert::get_state - unknown address ", adr);
      break;
  }
  return bv;

}

bert_fsm_state bert::get_fsm(void)
{
  return FSM;
}

bvec bert::get_poly(void)
{
  return prbs.get_poly();
}

int bert::get_metrics(void)
{
  return metrics;
}

bert_fsm_state bert::get_output(void)
{
  return y0;
}

void bert::clear_counters(void)
{
  cnt_bit = 0;
  cnt_err = 0;
  acc_cnt_bit = 0;
  acc_cnt_err = 0;
  return;
}

ivec bert::process(const bvec &ce, const bmat &x)
{
  ivec y;
  bmat bm;
  bvec bv;
  int N;

  debug(3) << "bert::process FSM =" << FSM << "\n ce=" << ce << "\n x=" << x << endl;

  if(x.cols() - symbol_size)
  {
    throw sim_exception(" bert::process - x.cols() != symbol_size ", x.cols() - symbol_size);
  }
  if(x.rows() - ce.length())
  {
    throw sim_exception(" bert::process - x.rows() != ce.length() ", x.rows() - ce.length());
  }
  if(symbol_size <= 0)
  {
    throw sim_exception(" bert::process - symbol_size <= 0 ", symbol_size);
  }

  N = ce.length();
  y.set_size(N);

  for(int i = 0; i < N; i++)
  {
    if(bool(ce[i]))
    {
      switch(FSM)
      {
        case BERT_FSM_RESET:
          cnt_bit = 0;
          cnt_err = 0;
          acc_cnt_bit = 0;
          acc_cnt_err = 0;
          bit_reloaded = 0;
          bit_preset = 0;
          metrics = -1;
          break;

        case BERT_FSM_RELOAD: // get data to load PRBS
          lsrx.process(one, x.get_rows(i, i)); // must be bmat , thus range
          bit_reloaded += symbol_size;
          metrics = -1;
          if(bit_reloaded >= prbs.get_length())
          {
            bv = lsrx.get_state();
            // bv[s(n+2)|s(n+1)|s(n)] - symbols are right shifted
            // but output of lfsr must be reversed to have continuous bit history
            prbs.set_state(bv.left(prbs.get_length()));
            bit_preset = 0;
            bit_reloaded = 0;
            FSM = BERT_FSM_PRESET;
          }
          break;

        case BERT_FSM_PRESET:
          lsrx.process(one, x.get_rows(i, i));
          lsry.process(one, prbs.generate(one));
          bit_preset += symbol_size;
          if(bit_preset >= lsrx.get_length())
          {
            bit_preset = 0;
            bit_reloaded = 0;
            cnt_bit = 0;   // reset when read
            cnt_err = 0;   // reset when read
            metrics = sum(to_ivec((lsrx.get_state() + lsry.get_state())));
            debug(3) << "lsrx.get_state() =" << lsrx.get_state() << endl;
            debug(3) << "lsry.get_state() =" << lsry.get_state() << endl;
            debug(3) << "           xor() =" << lsrx.get_state() + lsry.get_state() << endl;
            debug(3) << "   to_ivec(xor())=" << to_ivec((lsrx.get_state() + lsry.get_state())) << endl;
            debug(3) << "sum(to_ivec(xor )=" << sum(to_ivec((lsrx.get_state() + lsry.get_state()))) << endl;
            if(metrics > T2)
            {
              FSM = BERT_FSM_RELOAD;
            }
            else if(metrics > T0)
            {
              FSM = BERT_FSM_nSYNC;
            }
            else
            {
              FSM = BERT_FSM_SYNC;
            }
          }
          break;

        case BERT_FSM_nSYNC:
          lsrx.process(one, x.get_rows(i, i));
          lsry.process(one, prbs.generate(one));
          metrics = sum(to_ivec((lsrx.get_state() + lsry.get_state())));
          if(metrics > T2)
          {
            FSM = BERT_FSM_RELOAD;
          }
          else if(metrics <= T0)
          {
            FSM = BERT_FSM_SYNC;
          }
          break;

        case BERT_FSM_SYNC:
          lsrx.process(one, x.get_rows(i, i));
          bm = prbs.generate(one);
          lsry.process(one, bm);
          metrics = sum(to_ivec((lsrx.get_state() + lsry.get_state())));
          cnt_bit += symbol_size;
          cnt_err += sum(to_ivec((x.get_row(i) + bm.get_row(0))));
          acc_cnt_bit += symbol_size;
          acc_cnt_err += sum(to_ivec((x.get_row(i) + bm.get_row(0))));
          if(metrics > T1)
          {
            FSM = BERT_FSM_nSYNC;
          }
          break;

        default:
          throw sim_exception(" bert::process - unknown FSM state ", FSM);
      }
      y0 = FSM;
    }
    y[i] = y0;
  }
  debug(3) << "bert::process \n y=" << y << endl;
  return (y);

}


} // namespace SIM