/*xxx
* \brief - SIM::cofdm_map - map data as QAM codes into QAM carriers, boolean pilots into BPSK carriers
* \author maki
*/

#include "sim\sim_cofdm_map.hpp"
#include "sim\_sim_exception.hpp"
#include "__debug_level.h"
#include "debug.hpp"

namespace SIM
{

void cofdm_map::set_NFFT(int n)
{
  try
  {
    NFFT = n;
    y0.set_size(NFFT);
  }
  catch(...)
  {
    throw sim_exception("cofdm_map::set_NFFT - bad size", n);
  }
}

void cofdm_map::set_PA(double pa)
{
  PA = pa;
}

void cofdm_map::set_qam_size(int m)
{
  qammod.set_size(m);
}

void cofdm_map::set_data_carriers(ivec dc_indx)
{
  if(dc_indx.length() < NFFT)
  {
    if(itpp::min(dc_indx) >= 0)
    {
      if(itpp::max(dc_indx) < NFFT)
      {
        data_carriers = dc_indx;
      }
      else
      {
        throw sim_exception("cofdm_map::set_data_carriers - max(dcindx) > NFFT=", NFFT);
      }
    }
    else
    {
      throw sim_exception("cofdm_map::set_data_carriers - min(dcindx) <  0 ");
    }

  }
  else
  {
    throw sim_exception("cofdm_map::set_data_carriers - bad dcindx.size() > NFFT=", NFFT);
  }
}

void cofdm_map::set_pilots_carriers(ivec pc_indx)
{
  if(pc_indx.length() < NFFT)
  {
    if(itpp::min(pc_indx) >= 0)
    {
      if(itpp::max(pc_indx) < NFFT)
      {
        pilots_carriers = pc_indx;
      }
      else
      {
        throw sim_exception("cofdm_map::set_pilots_carriers - max(pcindx) > NFFT=", NFFT);
      }
    }
    else
    {
      throw sim_exception("cofdm_map::set_pilots_carriers - min(dcindx) <  0 ");
    }
  }
  else
  {
    throw sim_exception("cofdm_map::set_pilots_carriers - bad pcindx.size() > NFFT=", NFFT);
  }
}

void cofdm_map::set_zero_carriers(ivec zc_indx)
{
  if(zc_indx.length() < NFFT)
  {
    if(itpp::min(zc_indx) >= 0)
    {
      if(itpp::max(zc_indx) < NFFT)
      {
        zero_carriers = zc_indx;
      }
      else
      {
        throw sim_exception("cofdm_map::set_zero_carriers - max(zcindx) > NFFT=", NFFT);
      }
    }
    else
    {
      throw sim_exception("cofdm_map::set_zero_carriers - min(zcindx) <  0 ");
    }
  }
  else
  {
    throw sim_exception("cofdm_map::set_zero_carriers - bad zcindx.size() > NFFT=", NFFT);
  }

}

void cofdm_map::set_data(ivec x)
{
  cvec qv;
  bvec ce;
  int K = x.length();
  int i;

  ce.set_length(K);
  ce.ones();

  debug(3) << "cofdm_map::set_data data_carriers = " << data_carriers << endl;
  debug(3) << "cofdm_map::set_data x = " << x << endl;

  if (K != data_carriers.length())
  {
    throw sim_exception("cofdm_map::set_data - x.size() <> data_carriers.size()=", data_carriers.length());
  }
  
  // zero all carriers
  y0 = to_cvec(zeros(NFFT), zeros(NFFT));
  // set data carriers
  qv = qammod.process(ce, x);
  for(i = 0; i < K; i++)
  {
    y0(data_carriers(i)) = qv(i);
  }
  debug(3) << "cofdm_map::set_data y0 = " << y0 << endl;
}

void cofdm_map::set_pilots(bvec x)
{
  int K = x.length();
  complex<double> p1(PA , 0.0);
  complex<double> p0(-PA , 0.0);
  int L = zero_carriers.length();
  complex<double> c0(0.0 , 0.0);
  int i;

  debug(3) << "cofdm_map::set_pilots pilots_carriers = " << pilots_carriers << endl;
  debug(3) << "cofdm_map::set_pilots zero_carriers = " << zero_carriers << endl;
  debug(3) << "cofdm_map::set_pilots x = " << x << endl;

  if (K != pilots_carriers.length())
  {
    throw sim_exception("cofdm_map::set_pilots - x.size() <> pilots_carriers.size()=", pilots_carriers.length());
  }

  for(i = 0; i < K; i++)
  {
    y0(pilots_carriers(i)) = x(i) ? p1 : p0;
  }

  for(i = 0; i < L; i++)
  {
    y0(zero_carriers(i)) = c0;
  }

  debug(3) << "cofdm_map::set_pilots y0 = " << y0 << endl;

}

// data and pilots must be set with set_data, set_pilots
cvec cofdm_map::get_carriers()
{
  return (y0);
}

cmat cofdm_map::process(const bvec &ce, const imat &x)
{
  cmat y;
  int N;
  int K = data_carriers.length();
  int L = pilots_carriers.length();

  debug(3) << "cofdm_map::process ce = " << ce << endl;
  debug(3) << "cofdm_map::process  x = " << x << endl;

  N = ce.length();
  y.set_size(N, NFFT);

  if (x.rows() != N)
  {
    throw sim_exception("cofdm_map::process - ce.size <> x.rows", x.rows());
  }
  if (x.cols() != pilots_carriers.length() + data_carriers.length())
  {
    throw sim_exception("cofdm_map::process - x.cols <> pilots_carriers.length() + data_carriers.length() ", x.cols());
  }

  for (int i = 0; i < N; i++)
  {
    if (ce[i])
    {
      if (K)
      {
        cofdm_map::set_data(x.get_row(i).right(K));
      }
      if (L)
      {
        cofdm_map::set_pilots(to_bvec(x.get_row(i).left(L)));
      }
    }
    y.set_row(i, y0);
  }

  debug(3) << "cofdm_dem::process  y = " << y << endl;

  return (y);
}

} // namespace SIM