/*xxx
* \brief - SIM::src  - sample rate converter with sinc reconstruction filter
* \author maki
*/

#include "sim\sim_src.hpp"
#include "sim\_sim_exception.hpp"
#include "__debug_level.h"
#include "debug.hpp"

namespace SIM
{

void src::set_size(int n)
{
  if(n <= 0)
  {
    throw sim_exception("src::set_size - bad size", n);
  }
  // delete old x_cb
  if(NTaps > 0)
  {
    delete [] p_x_cb;
  }
  NTaps = n;
  p_x_cb = new double[NTaps];
  // reset buffer and pointer
  for(int i = 0; i < NTaps; i++) p_x_cb[i] = 0.0;
  x_cb_i = 0;
  return;
}

void src::set_win_type(int w)
{
  switch(w)
  {
    case WIN_RECT:
    case WIN_HANNING: // Hanning
    case WIN_KAISER_BESSEL: // Kaiser-Besell
    case WIN_FLAT_TOP:// Flat-Top
      WIN = (win_type)w;
      break;
    default: // 0 and others rectangular
      throw sim_exception("src::set_win_type - bad type", w);
      break;
  }
  return;
}

void src::set_output(double yout)
{
  y0 = yout;
}

int src::get_size(void)
{
  return (NTaps);
}

int src::get_win_type(void)
{
  return (WIN);
}

double src::get_output(void)
{
  return (y0);
}

double src::sinc(double x)
{
  double y;

  if(x == 0.0)
  {
    y = 1.0;
  }
  else
  {
    y = sin(pi * x) / (pi * x);
  }
  return (y);
}

// calculate casual sinc response for nt - normalized fractional time 
double src::h(double nt)
{
  double y, yw;
  double dTN, TN;
  double  w;
  double scale_cor;


  TN = (double)(NTaps - 1);       // normalized impulse response time (N-1)
  dTN = TN / 2;                   // casual delay (N-1)/2
  if((nt < TN) && (nt >= 0.0))
  {
    y = sinc(nt - dTN);
  }
  else
  {
    y = 0.0;
  }

  switch(WIN)
  {
    case WIN_HANNING: // Hanning
      w = 0.5 * (1 - cos(2 * pi * nt / TN));
      break;
    case WIN_KAISER_BESSEL: // Kaiser-Besell
      w = (1 - 1.24 * cos(2 * pi * nt / TN) + 0.244 * cos(4 * pi * nt / TN) - 0.00305 * cos(6 * pi * nt / TN)) / 2.48;
      scale_cor = 1 / 1.002843;
      w = w * scale_cor;
      break;
    case WIN_FLAT_TOP:// Flat-Top
      w = (1 - 1.93 * cos(2 * pi * nt / TN) + 1.29 * cos(4 * pi * nt / TN) - 0.388 * cos(6 * pi * nt / TN) - 0.0322 * cos(8 * pi * nt / TN)) / 4.46;
      scale_cor = 1 / 1.0259641;
      w = w * scale_cor;
      break;
    default: // 0 and others rectangular
      w = 1.0;
      break;
  }
  yw = y * w;
  return(yw);
}

void src::xcb(double x_new)
{
  int p;

  p = x_cb_i;       					// load cb-index
  p_x_cb[p] = x_new;       				// store new sample
  p++;
  if(p == NTaps)
  {
    p = 0;
  }
  x_cb_i = p;      					// store updated cb-index
  return ;

}

// calculate response for the samples in cb and fraction time tau
double src::re_samp(double tau)
{
  int p;
  int k;
  double sum, h_t, sample;


  p = (x_cb_i) - 1;       		// index of the latest sample 
  sum = 0.0;
  for(k = 0; k < NTaps; k++)
  {
    if(p < 0)                 // circular buffer
    {
      p = NTaps - 1;
    }
    h_t = h(tau + k);         // reconstruction of impulse contribution, k+tau represents t=k*Ti+tau
    sample = p_x_cb[p];	      // samples
    sum = sum + h_t * sample; // reconstruction of sample contribution 
    p = p - 1;      				  // go to older sample
  }
  return (sum);

}


vec src::process(const bmat &ceio, const mat &xt)
{
  vec y;
  int K;

  debug(3) << "src::process ceio = " << ceio << endl;
  debug(3) << "src::process   xt = " << xt << endl;

  if(ceio.cols() != 2)
  {
    throw sim_exception("src::process - ceio.cols() <> 2 ", ceio.cols());
  }
  if(xt.cols() != 2)
  {
    throw sim_exception("src::process - xt.cols() <> 2 ", xt.cols());
  }
  if(xt.rows() != ceio.rows())
  {
    throw sim_exception("src::process - xt.rows() <> ceio.rows() ", xt.rows());
  }

  K = ceio.rows();
  y.set_length(K);
  for(int i = 0; i < K; i++)
  {
    // cei - input sample to x_cb
    if(ceio(i, 0))
    {
      xcb(xt(i, 0));
    }
    // ceo - re_samp for tau
    if(ceio(i, 1))
    {
      y0 = re_samp(xt(i, 1));
    }
    y(i) = y0;
  }
  debug(3) << "src::process y=" << y << endl;
  return (y);
}

} // namespace SIM